﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace BeginAcceptTest001.Server
{
    internal class TakeClientQueue
    {
        internal System.Text.Encoding Encoding { get; private set; }
        
        private const int LIMIT_CONCURRENCY = 3;
        private Queue<System.Net.Sockets.TcpClient> clientQueue;
        private object queueLock = new object();
        private System.Threading.ManualResetEvent takenSignal;
        private System.Threading.CancellationToken cancelToken;
        private bool isStart;

        internal TakeClientQueue(System.Threading.CancellationToken token, System.Text.Encoding enc)
        {
            cancelToken = token;
            Encoding = enc;
            isStart = false;
        }

        internal void AddQueue(System.Net.Sockets.TcpClient client)
        {
            if (!isStart)
                throw new InvalidOperationException(
                    "AddQueue メソッドを呼び出すことはできません(Start メソッドが呼び出されていないか、終了が要求されています)。");

            lock (queueLock)
            {
                clientQueue.Enqueue(client);
                Console.WriteLine("{0} client をキューに登録しました。", DateTime.Now);
            }
            checkQueue();
        }

        internal void Start()
        {
            System.Threading.Semaphore pool = null;
            List<System.Threading.Tasks.Task> taskList = null;
            object taskListLock = new object();

            try
            {
                clientQueue = new Queue<System.Net.Sockets.TcpClient>();
                takenSignal = new System.Threading.ManualResetEvent(false); // 非シグナル状態で作成
                pool = new System.Threading.Semaphore(LIMIT_CONCURRENCY, LIMIT_CONCURRENCY);
                taskList = new List<System.Threading.Tasks.Task>();
                isStart = true;

                while (true)
                {
                    Console.WriteLine("{0} キューのセット待ち", DateTime.Now);
                    takenSignal.WaitOne();  // キューにセットされるまで待機する
                    Console.WriteLine("{0} キューのセット待ち通過", DateTime.Now);
                    Console.WriteLine("{0} セマフォ待ち", DateTime.Now);
                    pool.WaitOne();         // セマフォ待機(taskの並行処理数)
                    Console.WriteLine("{0} セマフォ待ち通過", DateTime.Now);

                    if (cancelToken.IsCancellationRequested)     // 終了要求時の対応
                        break;

                    // 別スレッドで doProcess() を動かすようにセット
                    System.Net.Sockets.TcpClient client;
                    lock (queueLock)
                    {
                        client = clientQueue.Dequeue(); // キューから取り出し
                        Console.WriteLine("{0} client をキューから取り出しました。", DateTime.Now);
                    }
                    var task = new System.Threading.Tasks.Task(() => doProcess(client, cancelToken), cancelToken);
                    lock (taskListLock)
                    {
                        taskList.Add(task);
                        Console.WriteLine("task数: {0}", taskList.Count);
                    }
                    //スレッド終了時に TcpClient のクローズとセマフォの開放を行うようにセット
                    task.ContinueWith((t) =>
                    {
                        client.Close();

                        //if (!isServiceEnd)
                        pool.Release();
                        lock (taskListLock)
                        {
                            taskList.Remove(task);
                            Console.WriteLine("task数: {0}", taskList.Count);
                        }
                    });
                    task.Start();

                    checkQueue();   // キューの状態をチェックしてシグナルをセットする
                }
            }
            finally
            {
                lock (queueLock)
                {
                    while (clientQueue.Count != 0)
                    {   // キューに接続要求が残っていたら、後始末
                        var client = clientQueue.Dequeue();
                        Console.WriteLine("{0} {1} からの接続要求を閉じます", DateTime.Now,
                            ((System.Net.IPEndPoint)client.Client.RemoteEndPoint).Address);
                        client.Close();     // キューにセットされている接続要求をクローズ
                    }
                }

                System.Threading.Tasks.Task[] tasks;
                lock (taskListLock)
                {
                    tasks = taskList.ToArray();
                }
                System.Threading.Tasks.Task.WaitAll(tasks);
                Console.WriteLine("{0} 子タスク・スレッドの終了を検知しました。",
                    DateTime.Now.ToString());

                if (pool != null)
                    pool.Close();

                if (takenSignal != null)
                {
                    takenSignal.Dispose();
                    takenSignal = null;
                }
            }

        }

        internal void Stop()
        {
            if (!isStart)
                throw new InvalidOperationException("Start メソッドより前に Stop メソッドを呼び出すことはできません。");

            takenSignal.Set();
            isStart = false;
        }

        private void checkQueue()
        {
            lock (queueLock)
            {
                // キャンセル要求が来ているか、キューが空でなかったらシグナル状態へ
                if (cancelToken.IsCancellationRequested || clientQueue.Count != 0)
                    takenSignal.Set();
                else
                    takenSignal.Reset();
            }
        }

        private void doProcess(System.Net.Sockets.TcpClient client, System.Threading.CancellationToken token)
        {
            System.Net.Sockets.NetworkStream stream = null;
            try
            {
                // ネットワークストリームを取得
                stream = client.GetStream();
                sendData(stream, "Server ready");

                var rString = receiveData(stream);

                token.ThrowIfCancellationRequested();   // キャンセル通知の確認 & キャンセル実行

                // サーバー側処理の代わりに15秒待つ
                //(並行処理の様子の観察に都合が良い)
                System.Threading.Thread.Sleep(5000);
                token.ThrowIfCancellationRequested();   // キャンセル通知の確認 & キャンセル実行

                System.Threading.Thread.Sleep(5000);
                token.ThrowIfCancellationRequested();   // キャンセル通知の確認 & キャンセル実行

                System.Threading.Thread.Sleep(5000);
                token.ThrowIfCancellationRequested();   // キャンセル通知の確認 & キャンセル実行

                sendData(stream, rString);
            }
            catch (System.IO.IOException e)
            {
                if (e.InnerException.GetType() == typeof(System.Net.Sockets.SocketException) &&
                    (e.InnerException as System.Net.Sockets.SocketException).ErrorCode == 10054)
                    Console.WriteLine("{0} クライアントが切断していたためデータを送信できませんでした。",
                        DateTime.Now.ToString());
                else
                    throw;
            }
            catch (OperationCanceledException)
            {   // キャンセル実施をキャッチ
                Console.WriteLine("{0} 子タスクをキャンセルしました。", DateTime.Now.ToString());
            }
            finally
            {
                if (stream != null)
                {
                    Console.WriteLine("{0} {1} port[{2}] クライアントが切断しました。",
                        DateTime.Now.ToString(),
                        ((System.Net.IPEndPoint)client.Client.LocalEndPoint).Address.ToString(),
                        ((System.Net.IPEndPoint)client.Client.LocalEndPoint).Port.ToString());
                    stream.Close();
                }
            }
        }

        private string receiveData(System.IO.Stream stream)
        {
            var receiveMessage = "";
            using (var memStream = new System.IO.MemoryStream())
            {
                var rBuff = new byte[256];
                int rSize;
                do
                {
                    // データの一部を受信する
                    rSize = stream.Read(rBuff, 0, rBuff.Length);
                    // rSize が 0 のときにはクライアントが切断したと判断
                    if (rSize == 0)
                    {
                        Console.WriteLine("クライアントが切断しました。");
                        throw new ServerServiceException("ストリームの読み出しの際、クライアントが切断していました。",
                            ServerServiceException.Errors.ConnectionClose);
                    }
                    // 受信したデータを蓄積する
                    memStream.Write(rBuff, 0, rSize);
                } while ((stream as System.Net.Sockets.NetworkStream).DataAvailable);

                // 受信したデータを文字列に変換
                receiveMessage = Encoding.GetString(memStream.ToArray());
                memStream.Close();
            }

            Console.WriteLine("{0} [C] {1}", DateTime.Now.ToString(), receiveMessage);
            return receiveMessage;
        }

        private void sendData(System.IO.Stream stream, string str)
        {
            // 文字列をバイト配列へ
            var data = Encoding.GetBytes(str);
            stream.Write(data, 0, data.Length);
            Console.WriteLine("{0} [S] {1}", DateTime.Now.ToString(), str);
        }
    }
}
