﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace BeginAcceptTest001.Server
{
    public class ServerService
    {
        public string Ipv4Addr { get; set; }
        public string Ipv6Addr { get; set; }
        public int Port { get; set; }
        public System.Text.Encoding Encoding { get; set; }

        private System.Net.Sockets.TcpListener v4Listner = null;
        private System.Net.Sockets.TcpListener v6Listner = null;
        private System.Threading.Tasks.Task taskListen = null;
        private volatile bool isServiceEnd = false;
        private bool isV4Connected = true;
        private bool isV6Connected = true;
        private System.Threading.ManualResetEvent clientConnected =
            new System.Threading.ManualResetEvent(false);
        private System.Threading.ManualResetEvent ListenEnded;
        private int acceptWaitCounter = 0;
        private object acceptWaitCounterLock = new object();
        private Action notifyCancel;
        private TakeClientQueue takeQueue;

        public ServerService()
        {
        }

        public void StartService()
        {
            if (string.IsNullOrEmpty(Ipv4Addr))
                throw new ServerServiceException("IPv4アドレスが設定されていません。",
                    ServerServiceException.Errors.IpAddressError);
            if (string.IsNullOrEmpty(Ipv6Addr))
                throw new ServerServiceException("IPv6アドレスが設定されていません。",
                    ServerServiceException.Errors.IpAddressError);
            if (Port < 1024)
                throw new ServerServiceException("ポート番号に予約ポートが設定されています。",
                    ServerServiceException.Errors.PortOutOfRange);
            if (Encoding == null)
                throw new ServerServiceException("Encoding がセットされていません。",
                    ServerServiceException.Errors.EncodingNotSet);

            // 指定された IP アドレスが自ホストのものであることを確認する
            var hostName = System.Net.Dns.GetHostName();
            var addrList = System.Net.Dns.GetHostAddresses(hostName);
            try
            {
                if (!addrList.Contains(System.Net.IPAddress.Parse(Ipv4Addr)))
                    throw new ServerServiceException("指定されたIPv4アドレスは自ホストのアドレスではありません。",
                        ServerServiceException.Errors.IpAddressError);
            }
            catch (FormatException)
            {
                throw new ServerServiceException("指定されたIPv4アドレスは有効なアドレスではありません。",
                    ServerServiceException.Errors.IpAddressError);
            }
            try
            {
                if (!addrList.Contains(System.Net.IPAddress.Parse(Ipv6Addr)))
                    throw new ServerServiceException("指定されたIPv6アドレスは自ホストのアドレスではありません。",
                        ServerServiceException.Errors.IpAddressError);
            }
            catch
            {
                throw new ServerServiceException("指定されたIPv6アドレスは有効なアドレスではありません。",
                    ServerServiceException.Errors.IpAddressError);
            }

            taskListen = System.Threading.Tasks.Task.Factory.StartNew(() =>
            {
                startListen();
            });

        }

        public void StopService()
        {
            Console.WriteLine("{0} サービス終了が要求されました。", DateTime.Now);
            ListenEnded = new System.Threading.ManualResetEvent(false); // ListenEnded を非シグナル状態で作成
            isServiceEnd = true;
            if (notifyCancel != null)
                notifyCancel();
            endListen();
            taskListen.Wait();
            Console.WriteLine("{0} Listen スレッドの 終了を検知しました。",
                DateTime.Now.ToString());
        }

        private void startListen()
        {
            var tokenSource = new System.Threading.CancellationTokenSource();
            notifyCancel = new Action(() => tokenSource.Cancel());
            var token = tokenSource.Token;

            Console.WriteLine("{0} Listen スレッドが開始しました。",
                DateTime.Now.ToString());

            takeQueue = new TakeClientQueue(token, Encoding);
            System.Threading.Tasks.Task takeTask = null;

            try
            {
                takeTask = new System.Threading.Tasks.Task(() => takeQueue.Start(), token);
                takeTask.Start();

                var ipv6Address = System.Net.IPAddress.Parse(Ipv6Addr);
                v6Listner = new System.Net.Sockets.TcpListener(ipv6Address, Port);
                v6Listner.Start();
                Console.WriteLine("{0} {1} Port[{2}]の Listen を開始しました。",
                    DateTime.Now.ToString(), ipv6Address.ToString(), Port);
                var ipv4Address = System.Net.IPAddress.Parse(Ipv4Addr);
                v4Listner = new System.Net.Sockets.TcpListener(ipv4Address, Port);
                v4Listner.Start();
                Console.WriteLine("{0} {1} Port[{2}]の Listen を開始しました。",
                    DateTime.Now.ToString(), ipv4Address.ToString(), Port);

                while (true)
                {
                    clientConnected.Reset();

                    if (isServiceEnd)     // 終了要求時の対応
                        break;

                    if (isV6Connected)
                    {   // V6接続の Callback が行われたら、新たなV6接続待ちを行う
                        isV6Connected = false;
                        Console.WriteLine("{0} ipv6 の接続を待ちます", DateTime.Now.ToString());
                        lock (acceptWaitCounterLock)
                        {
                            acceptWaitCounter++;
                        }
                        v6Listner.BeginAcceptTcpClient(new AsyncCallback(acceptCallback), v6Listner);
                    }

                    if (isV4Connected)
                    {   // V4接続の Callback が行われたら、新たなV4接続待ちを行う
                        isV4Connected = false;
                        Console.WriteLine("{0} ipv4 の接続を待ちます", DateTime.Now.ToString());
                        lock (acceptWaitCounterLock)
                        {
                            acceptWaitCounter++;
                        }
                        v4Listner.BeginAcceptTcpClient(new AsyncCallback(acceptCallback), v4Listner);
                    }

                    Console.WriteLine("{0} 接続待ち待機", DateTime.Now);
                    clientConnected.WaitOne();  // 接続が行われるまで待機する
                    Console.WriteLine("{0} 接続待ち通過", DateTime.Now);
                }
            }
            finally
            {
                Console.WriteLine("{0} Accept 終了待ち待機", DateTime.Now);
                ListenEnded.WaitOne();  // 別スレッドの Accept 待ちが終了するのを待つ
                Console.WriteLine("{0} Accept 終了待ち通過", DateTime.Now);

                Console.WriteLine("{0} TakeQueue 終了待ち待機", DateTime.Now);
                if (takeTask != null)
                    takeTask.Wait();
                Console.WriteLine("{0} TakeQueue 終了待ち通過", DateTime.Now);

                clientConnected.Close();

                tokenSource.Dispose();
            }

        }

        private void endListen()
        {
            Console.WriteLine("{0} {1} Port[{2}]の Listen を終了します。",
                DateTime.Now.ToString(), Ipv6Addr, Port);
            if (v6Listner != null)
                v6Listner.Stop();
            Console.WriteLine("{0} {1} Port[{2}]の Listen を終了します。",
                DateTime.Now.ToString(), Ipv4Addr, Port);
            if (v4Listner != null)
                v4Listner.Stop();

            takeQueue.Stop();
        }

        private void acceptCallback(IAsyncResult ar)
        {
            Console.WriteLine("{0} Callback が呼び出されました。", DateTime.Now);
            if (!isServiceEnd)
            {
                if (ar.IsCompleted)
                {
                    var listner = (System.Net.Sockets.TcpListener)ar.AsyncState;
                    var client = listner.EndAcceptTcpClient(ar);
                    Console.WriteLine("{0} {1} port[{2}] クライアントが接続しました。",
                        DateTime.Now.ToString(),
                        ((System.Net.IPEndPoint)listner.LocalEndpoint).Address.ToString(),
                        ((System.Net.IPEndPoint)listner.LocalEndpoint).Port.ToString());

                    // client をキューにセット
                    takeQueue.AddQueue(client);

                    if (((System.Net.IPEndPoint)listner.LocalEndpoint).Address.AddressFamily ==
                        System.Net.Sockets.AddressFamily.InterNetwork)
                    {
                        isV4Connected = true;
                    }
                    else
                    {
                        isV6Connected = true;
                    }
                }
            }

            lock (acceptWaitCounterLock)
            {
                acceptWaitCounter--;
            }
            Console.WriteLine("{0} 接続待機のシグナルをセット", DateTime.Now);
            clientConnected.Set();  // 接続待機をシグナル状態にする
            checkEndListen();
        }

        private void checkEndListen()
        {
            if (isServiceEnd && acceptWaitCounter <= 0)
            {
                ListenEnded.Set();  // 終了時に Accept 待ちがなくなったらシグナル状態にする
                Console.WriteLine("{0} ListenEnded セット", DateTime.Now);
            }
        }
    }
}
