﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Threading;

namespace DateTimePicker.UserControls.Behaviors
{
    internal class TextBoxHelper
    {
        /// <summary>
        /// フォーカス取得時のテキスト全選択の有無を設定します(true で全選択)。
        /// </summary>
        public static readonly DependencyProperty IsFocusSelectProperty = DependencyProperty.RegisterAttached(
            "IsFocusSelect", typeof(bool), typeof(TextBoxHelper),
            new UIPropertyMetadata(false, IsFocusSelectChanged));

        private static void IsFocusSelectChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var textBox = (TextBox)sender;
            if (textBox == null) return;

            // 設定値を見てイベントを登録・削除
            var newValue = (bool)e.NewValue;
            var oldValue = (bool)e.OldValue;
            if (newValue == oldValue) return;
            if (oldValue)
            {
                textBox.GotFocus -= textBox_GotFocus;
            }
            if (newValue)
            {
                textBox.GotFocus += textBox_GotFocus;
            }
        }

        private static void textBox_GotFocus(object sender, RoutedEventArgs e)
        {
            var textBox = (TextBox)e.OriginalSource;
            if (textBox == null) return;
            // 非同期で全選択処理を実行する
            Action selectAction = textBox.SelectAll;
            Dispatcher.CurrentDispatcher.BeginInvoke(selectAction, DispatcherPriority.Background);
        }

        [AttachedPropertyBrowsableForType(typeof(TextBox))]
        public static bool GetIsFocusSelect(DependencyObject obj)
        {
            return (bool)obj.GetValue(IsFocusSelectProperty);
        }

        [AttachedPropertyBrowsableForType(typeof(TextBox))]
        public static void SetIsFocusSelect(DependencyObject obj, bool value)
        {
            obj.SetValue(IsFocusSelectProperty, value);
        }
    }
}
