﻿using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using DateTimePicker.UserControls.ViewModels;

namespace DateTimePicker.UserControls
{
    /// <summary>
    /// DateTimePicker.xaml の相互作用ロジック
    /// </summary>
    public partial class DateTimePicker : UserControl
    {
        public DateTimePicker()
        {
            InitializeComponent();

            // DatePicker へのバインド
            var datePickerBind = new Binding("SelectedDate");
            datePickerBind.Source = DatePicker;
            datePickerBind.Mode = BindingMode.TwoWay;
            this.SetBinding(SelectedDateProperty, datePickerBind);

            // ViewModel の Hour へのバインド
            var hourBind = new Binding("Hour");
            hourBind.Source = (DateTimePickerViewModel)BasePanel.DataContext;
            hourBind.Mode = BindingMode.TwoWay;
            this.SetBinding(InputtedHourProperty, hourBind);

            // ViewModel の Minute へのバインド
            var minuteBind = new Binding("Minute");
            minuteBind.Source = (DateTimePickerViewModel)BasePanel.DataContext;
            minuteBind.Mode = BindingMode.TwoWay;
            this.SetBinding(InputtedMinuteProperty, minuteBind);

            // ViewModel の Second へのバインド
            var secondBind = new Binding("Second");
            secondBind.Source = (DateTimePickerViewModel)BasePanel.DataContext;
            secondBind.Mode = BindingMode.TwoWay;
            this.SetBinding(InputtedSecondProperty, secondBind);

            DatePicker.SelectedDate = DateTime.Now;
        }

        public static readonly DependencyProperty DateTimeValueProperty = DependencyProperty.Register(
                "DateTimeValue", typeof(DateTime), typeof(DateTimePicker), 
                new FrameworkPropertyMetadata   // メタデータ
                {
                    DefaultValue = DateTime.Now,   // デフォルト値
                    BindsTwoWayByDefault = true,
                    PropertyChangedCallback = new PropertyChangedCallback(onDateTimeValueChanged)
                });

        // 依存関係プロパティのラッパー
        public DateTime DateTimeValue
        {
            get { return (DateTime)GetValue(DateTimeValueProperty); }
            set { SetValue(DateTimeValueProperty, value); }
        }

        private static void onDateTimeValueChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            if ((DateTime)e.OldValue != (DateTime)e.NewValue)
            {
                var newDateTime = (DateTime)e.NewValue;
                userControl.InputtedHour = newDateTime.ToString("HH");
                userControl.InputtedMinute = newDateTime.ToString("mm");
                userControl.InputtedSecond = newDateTime.ToString("ss");
                userControl.SelectedDate = newDateTime;
                userControl.DateTimeString = newDateTime.ToString("F");
            }
        }

        public static readonly DependencyProperty SelectedDateFormatProperty = DependencyProperty.Register(
            "SelectedDateFormat", typeof(DatePickerFormat), typeof(DateTimePicker),
            new FrameworkPropertyMetadata   // メタデータ
            {
                DefaultValue = DatePickerFormat.Long,
                PropertyChangedCallback = new PropertyChangedCallback(onSelectedDateFormatChanged)
            });

        // 依存関係プロパティのラッパー
        public DatePickerFormat SelectedDateFormat
        {
            get { return (DatePickerFormat)GetValue(SelectedDateFormatProperty); }
            set { SetValue(SelectedDateFormatProperty, value); }
        }

        private static void onSelectedDateFormatChanged(DependencyObject sender,
            DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            userControl.DatePicker.SelectedDateFormat = (DatePickerFormat)e.NewValue;
        }

        public static readonly DependencyProperty IsDisplayedWeekProperty = DependencyProperty.Register(
            "IsDisplayedWeek", typeof(bool), typeof(DateTimePicker),
            new FrameworkPropertyMetadata   // メタデータ
            {
                DefaultValue = true,
                PropertyChangedCallback = new PropertyChangedCallback(onIsDisplayedWeekChanged)
            });

        // 依存関係プロパティのラッパー
        public bool IsDisplayedWeek
        {
            get { return (bool)GetValue(IsDisplayedWeekProperty); }
            set { SetValue(IsDisplayedWeekProperty, value); }
        }

        private static void onIsDisplayedWeekChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            setWeek(userControl, userControl.DateTimeValue);
        }

        public static readonly DependencyProperty DateTimeStringProperty = DependencyProperty.Register(
                "DateTimeString", typeof(string), typeof(DateTimePicker), 
                new FrameworkPropertyMetadata());   // メタデータ

        // 依存関係プロパティのラッパー
        public string DateTimeString
        {
            get { return (string)GetValue(DateTimeStringProperty); }
            private set { SetValue(DateTimeStringProperty, value); }
        }

        // 内部処理用の依存関係プロパティ(DatePicker コントロールとバインド)
        private static readonly DependencyProperty SelectedDateProperty = DependencyProperty.Register(
                "SelectedDate", typeof(DateTime), typeof(DateTimePicker), 
                new FrameworkPropertyMetadata   // メタデータ
                {
                    BindsTwoWayByDefault = true,
                    PropertyChangedCallback = new PropertyChangedCallback(onSelectedDateChanged)
                });

        private DateTime SelectedDate
        {
            get { return (DateTime)GetValue(SelectedDateProperty); }
            set { SetValue(SelectedDateProperty, value); }
        }

        private static void onSelectedDateChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            if ((DateTime)e.OldValue != (DateTime)e.NewValue)
            {
                var newdate = (DateTime)e.NewValue;
                var dateTime = string.Format("{0}/{1}/{2} {3}:{4}:{5}",
                    newdate.Year, newdate.Month, newdate.Day,
                    userControl.InputtedHour, userControl.InputtedMinute, userControl.InputtedSecond);
                userControl.DateTimeValue = DateTime.Parse(dateTime);
                setWeek(userControl, (DateTime)e.NewValue);
            }
        }

        // 画面表示の曜日をセット
        private static void setWeek(DateTimePicker userControl, DateTime date)
        {
            if (userControl.IsDisplayedWeek)
            {   
                userControl.DisplayWeek.Text = string.Format("({0})", date.ToString("ddd"));
            }
            else
            {
                userControl.DisplayWeek.Text = string.Empty;
            }
        }

        // 内部処理用の依存関係プロパティ(TextBox(Hour) コントロールとバインド)
        private static readonly DependencyProperty InputtedHourProperty = DependencyProperty.Register(
                "InputtedHour", typeof(string), typeof(DateTimePicker), 
                new FrameworkPropertyMetadata   // メタデータ
                {
                    DefaultValue = "00",    // デフォルト値
                    BindsTwoWayByDefault = true,
                    PropertyChangedCallback = new PropertyChangedCallback(onInputtedHourChanged)
                });

        private string InputtedHour
        {
            get { return (string)GetValue(InputtedHourProperty); }
            set { SetValue(InputtedHourProperty, value); }
        }

        private static void onInputtedHourChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            if ((string)e.OldValue != (string)e.NewValue)
            {
                var date = userControl.DateTimeValue;
                var dateTime = string.Format("{0}/{1}/{2} {3}:{4}:{5}",
                    date.Year, date.Month, date.Day,
                    (string)e.NewValue, userControl.InputtedMinute, userControl.InputtedSecond);
                userControl.DateTimeValue = DateTime.Parse(dateTime);
            }
        }

        // 内部処理用の依存関係プロパティ(TextBox(Minute) コントロールとバインド)
        private static readonly DependencyProperty InputtedMinuteProperty = DependencyProperty.Register(
                "InputtedMinute", typeof(string), typeof(DateTimePicker), 
                new FrameworkPropertyMetadata   // メタデータ
                {
                    DefaultValue = "00",    // デフォルト値
                    BindsTwoWayByDefault = true,
                    PropertyChangedCallback = new PropertyChangedCallback(onInputtedMinuteChanged)
                });

        private string InputtedMinute
        {
            get { return (string)GetValue(InputtedMinuteProperty); }
            set { SetValue(InputtedMinuteProperty, value); }
        }

        private static void onInputtedMinuteChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            if ((string)e.OldValue != (string)e.NewValue)
            {
                var date = userControl.DateTimeValue;
                var dateTime = string.Format("{0}/{1}/{2} {3}:{4}:{5}",
                    date.Year, date.Month, date.Day,
                    userControl.InputtedHour, (string)e.NewValue, userControl.InputtedSecond);
                userControl.DateTimeValue = DateTime.Parse(dateTime);
            }
        }

        // 内部処理用の依存関係プロパティ(TextBox(Second) コントロールとバインド)
        private static readonly DependencyProperty InputtedSecondProperty = DependencyProperty.Register(
                "InputtedSecond", typeof(string), typeof(DateTimePicker), 
                new FrameworkPropertyMetadata   // メタデータ
                {
                    DefaultValue = "00",    // デフォルト値
                    BindsTwoWayByDefault = true,
                    PropertyChangedCallback = new PropertyChangedCallback(onInputtedSecondChanged)
                });

        // 依存関係プロパティのラッパー
        private string InputtedSecond
        {
            get { return (string)GetValue(InputtedSecondProperty); }
            set { SetValue(InputtedSecondProperty, value); }
        }

        private static void onInputtedSecondChanged(DependencyObject sender, DependencyPropertyChangedEventArgs e)
        {
            var userControl = (DateTimePicker)sender;
            if ((string)e.OldValue != (string)e.NewValue)
            {
                var date = userControl.DateTimeValue;
                var dateTime = string.Format("{0}/{1}/{2} {3}:{4}:{5}",
                    date.Year, date.Month, date.Day,
                    userControl.InputtedHour, userControl.InputtedMinute, (string)e.NewValue);
                userControl.DateTimeValue = DateTime.Parse(dateTime);
            }
        }
    }
}
