﻿using System;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Text;

namespace IpV4V6AcceptTest.Client
{
    enum IpKinds
    {
        Undecided, IpV4, IpV6
    }

    class Client
    {
        static void Main(string[] args)
        {
            // 文字コードを UTF-8 とする
            var enc = Encoding.UTF8;

            // サーバー IP アドレス
            var ipv4 = "127.0.0.1";
            var ipv6 = "::1";
            // サーバーポート
            var serverPort = 10050;

            IpKinds selectedIp = IpKinds.Undecided;
            while (selectedIp == IpKinds.Undecided)
            {
                Console.WriteLine("接続する IP アドレスの種類を選択してください...(1: IPv4, 2:IPv6)");
                switch (Console.ReadLine())
                {
                    case "1":
                        selectedIp = IpKinds.IpV4;
                        break;
                    case "2":
                        selectedIp = IpKinds.IpV6;
                        break;
                    default:
                        Console.WriteLine("入力された値が不正です。");
                        break;
                }
            }

            var isSuccessClient = false;
            TcpClient client = null;
            try
            {
                // TcpClient を作成し、サーバーと接続する
                IPAddress serverAddr = null;
                if (selectedIp == IpKinds.IpV6)
                {
                    serverAddr = IPAddress.Parse(ipv6);
                    client = new TcpClient(AddressFamily.InterNetworkV6);
                }
                else
                {
                    serverAddr = IPAddress.Parse(ipv4);
                    client = new TcpClient(AddressFamily.InterNetwork);
                }

                client.Connect(serverAddr, serverPort);
                Console.WriteLine("{0} サーバーと接続しました。", DateTime.Now.ToLongTimeString());
                isSuccessClient = true;

                // ネットワークストリームを取得する
                using (var stream = client.GetStream())
                {
                    var welcome = receiveData(stream, enc);

                    var sendString = "abc0123あいうえお";
                    sendData(stream, sendString, enc);

                    var receiveString = receiveData(stream, enc);

                    stream.Close();
                }
            }
            catch (FormatException e)
            {
                Console.WriteLine(e.Message);
            }
            catch (SocketException e)
            {
                switch (e.ErrorCode)
                {
                    case 10061:
                        Console.WriteLine("{0} サーバーへの接続が拒否されました。",
                            DateTime.Now.ToLongTimeString());
                        break;
                    default:
                        Console.WriteLine("{0} ソケットエラー コード: {1}",
                            DateTime.Now.ToLongTimeString(), e.ErrorCode);
                        break;
                }
            }
            catch (IOException e)
            {
                if (e.InnerException.GetType() == typeof(SocketException))
                {
                    switch ((e.InnerException as SocketException).ErrorCode)
                    {
                        case 10053:
                            Console.WriteLine("{0} サーバーとの接続がサーバー側のタイムアウトにより切断されました。",
                                DateTime.Now.ToLongTimeString());
                            break;
                        case 10054:
                            Console.WriteLine("{0} サーバーとの接続がサーバー側で強制切断されました。",
                                DateTime.Now.ToLongTimeString());
                            break;
                        default:
                            Console.WriteLine("{0} ソケットエラー コード: {1}",
                                DateTime.Now.ToLongTimeString(),
                                (e.InnerException as SocketException).ErrorCode);
                            break;
                    }
                }
                else
                    throw;
            }
            catch (ConnectionCloseException)
            {
                Console.WriteLine("サーバーからの受信待ち中にサーバーが切断しました");
            }
            finally
            {
                if (isSuccessClient)
                {
                    client.Close();
                    Console.WriteLine("{0} 切断しました。", DateTime.Now.ToLongTimeString());
                }
            }

            Console.WriteLine("Enter キー押下で終了します。");
            Console.ReadLine();
        }

        static string receiveData(System.IO.Stream stream, System.Text.Encoding enc)
        {
            var receiveMessage = "";
            using (var memStream = new System.IO.MemoryStream())
            {
                var rBuf = new byte[256];
                int rSize;
                do
                {
                    // データの一部を受信する
                    rSize = stream.Read(rBuf, 0, rBuf.Length);
                    // rSize が 0 のときにはサーバーが切断したと判断
                    if (rSize == 0)
                    {
                        Console.WriteLine("{0} サーバーが切断しました。", DateTime.Now.ToLongTimeString());
                        throw new ConnectionCloseException("ストリームの読み出しの際、サーバーが切断していました。");
                    }
                    // 受信したデータを蓄積する
                    memStream.Write(rBuf, 0, rSize);
                } while ((stream as System.Net.Sockets.NetworkStream).DataAvailable);

                // 受信したデータを文字列に変換
                receiveMessage = enc.GetString(memStream.ToArray());
                memStream.Close();
            }

            Console.WriteLine("{0} [S] {1}", DateTime.Now.ToString(), receiveMessage);
            return receiveMessage;
        }

        static void sendData(System.IO.Stream stream, string str, System.Text.Encoding enc)
        {
            // 文字列をバイト配列へ
            var data = enc.GetBytes(str);
            stream.Write(data, 0, data.Length);
            Console.WriteLine("{0} [C] {1}", DateTime.Now.ToString(), str);
        }
    }
}
