﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;

namespace IpV4V6AcceptTest.Server
{
    class AsyncProducerConsumerCollection<T>
    {
        // キューへの投入を待機させるためのキュー(TResult の bool 型はダミー)
        // このキューにセットされるオブジェクトの最大数はプロデューサの稼働数
        private readonly Queue<TaskCompletionSource<bool>> _queueLimitTaskQueue;
        // 渡すデータのキュー
        private readonly Queue<T> _collection;
        // 渡すデータが無かった時に未完了状態で返した TaskCompletionSource オブジェクトのキュー
        private readonly Queue<TaskCompletionSource<T>> _waiting;

        // queueEntriesLimit が 0 のときにはキューへの投入制限を行わない(デフォルト値 0)
        public AsyncProducerConsumerCollection(int queueEntriesLimit = 0)
        {
            _collection = new Queue<T>();
            _waiting = new Queue<TaskCompletionSource<T>>();
            QueueEntriesLimit = queueEntriesLimit;
            _queueLimitTaskQueue = new Queue<TaskCompletionSource<bool>>();
        }

        public int QueueEntriesLimit { get; private set; }

        public void Add(T item)
        {
            Add(item, default(CancellationToken));
        }

        public void Add(T item, CancellationToken token)
        {
            TaskCompletionSource<bool> queueLimitTask = null;

            lock (_collection)
            {
                // キューへの投入制限値に達していた場合、待機用の TaskCompletionSource オブジェクトを生成
                if (QueueEntriesLimit != 0 &&
                    _collection.Count >= QueueEntriesLimit)
                {
                    queueLimitTask = new TaskCompletionSource<bool>();
                    _queueLimitTaskQueue.Enqueue(queueLimitTask);
                    System.Diagnostics.Trace.WriteLine(
                        string.Format("{0} データキュー登録待ちの数: {1}",
                        DateTime.Now.ToLongTimeString(), _queueLimitTaskQueue.Count));
                }
            }
            // キューへの投入待機用オブジェクトが生成されていた場合には完了まで待機する
            if (queueLimitTask != null)
            {
                queueLimitTask.Task.Wait();
            }
            // キャンセル受付後に進入してきたら登録せずにリターン
            if (token.WaitHandle != null && token.IsCancellationRequested)
            {
                disposeItem(item);
                return;
            }

            TaskCompletionSource<T> tcs = null;
            lock (_collection) // キュー操作の排他制御
            {
                if (_waiting.Count > 0)
                {
                    tcs = _waiting.Dequeue();
                }
                else
                {
                    _collection.Enqueue(item);
                    System.Diagnostics.Trace.WriteLine(
                        string.Format("{0} データキューの数: {1}",
                        DateTime.Now.ToLongTimeString(), _collection.Count));
                }
            }
            if (tcs != null)
            {
                // 未完了状態のオブジェクトにセットするとき
                // item のセットとともに、タスクの状態を「正常に完了」への遷移を試みる
                tcs.TrySetResult(item);
            }
        }

        public Task<T> Take()
        {
            return Take(default(CancellationToken));
        }

        public Task<T> Take(CancellationToken token)
        {
            if (token.WaitHandle != null)
            {
                token.Register(() => canceled());
            }

            lock (_collection)
            {
                if (_collection.Count > 0)
                {
                    // データのキューが空でなければキューから取り出す
                    return Task<T>.Factory.StartNew(() =>
                    {
                        var item = _collection.Dequeue();
                        System.Diagnostics.Trace.WriteLine(
                            string.Format("{0} データキューの数: {1}",
                            DateTime.Now.ToLongTimeString(), _collection.Count));
                        // キューへの投入待機用オブジェクトが生成されていた場合には一つを進行させる
                        queueLimitDequeue();
                        return item;
                    });
                }
                else
                {
                    // キューが空だったら、未完了状態の TaskCompletionSource から作成されるタスクを返す
                    var tcs = new TaskCompletionSource<T>();
                    _waiting.Enqueue(tcs);
                    return tcs.Task;
                }
            }
        }

        private void canceled()
        {
            lock (_collection)
            {
                while (_queueLimitTaskQueue.Count > 0)
                {
                    // キューへの投入待機を進行させる
                    queueLimitDequeue();
                }
                
                // キューに入っているデータをクリア
                while (_collection.Count > 0)
                {
                    disposeItem(_collection.Dequeue());
                    System.Diagnostics.Trace.WriteLine(
                        string.Format("{0} データキューの数: {1}",
                        DateTime.Now.ToLongTimeString(), _collection.Count));
                }

                while (_waiting.Count > 0)
                {
                    // 未完了状態の TaskCompletionSource オブジェクトを取り出し
                    var tcs = _waiting.Dequeue();
                    // Canceld 状態への遷移を試みる
                    tcs.TrySetCanceled();
                }
            }
        }

        private void queueLimitDequeue()
        {
            if (_queueLimitTaskQueue.Count != 0)
            {
                _queueLimitTaskQueue.Dequeue().TrySetResult(true);
                System.Diagnostics.Trace.WriteLine(
                    string.Format("{0} データキュー登録待ちの数: {1}",
                    DateTime.Now.ToLongTimeString(), _queueLimitTaskQueue.Count));
            }
        }

        private void disposeItem(T item)
        {
            if (item as IDisposable != null)
            {
                (item as IDisposable).Dispose();
            }
        }
    }
}
