﻿using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;

namespace MultiThreadAcceptTest2.Server
{
    class AsyncProducerConsumerCollection<T>
    {
        // キューへの投入を待機させるための TaskCompletionSource オブジェクト(TResult の bool 型はダミー)
        private TaskCompletionSource<bool> _queueLimitTask;
        // 渡すデータのキュー
        private readonly Queue<T> _collection;
        // 渡すデータが無かった時に未完了状態で返した TaskCompletionSource オブジェクトのキュー
        private readonly Queue<TaskCompletionSource<T>> _waiting;

        // queueEntriesLimit が 0 のときにはキューへの投入制限を行わない(デフォルト値 0)
        public AsyncProducerConsumerCollection(int queueEntriesLimit = 0)
        {
            _collection = new Queue<T>();
            _waiting = new Queue<TaskCompletionSource<T>>();
            _queueEntriesLimit = queueEntriesLimit;
            _queueLimitTask = null;
        }

        private readonly int _queueEntriesLimit;
        public int QueueEntriesLimit
        {
            get { return _queueEntriesLimit; }
        }

        public void Add(T item)
        {
            TaskCompletionSource<T> tcs = null;
            lock (_collection)
            {
                if (_waiting.Count > 0)
                {
                    tcs = _waiting.Dequeue();
                }
                else
                {
                    _collection.Enqueue(item);
                    System.Diagnostics.Trace.WriteLine(
                        string.Format("{0} データキューの数: {1}",
                        DateTime.Now.ToLongTimeString(), _collection.Count));
                    // キューへの投入制限値に達していた場合、待機用の TaskCompletionSource オブジェクトを生成
                    if (QueueEntriesLimit != 0 &&
                        _collection.Count >= QueueEntriesLimit)
                    {
                        _queueLimitTask = new TaskCompletionSource<bool>();
                    }
                }
            }
            if (tcs != null)
            {
                // 未完了状態のオブジェクトにセットするとき
                // item のセットとともに、タスクの状態を「正常に完了」への遷移を試みる
                tcs.TrySetResult(item);
            }
            // キューへの投入待機用オブジェクトが生成されていた場合には完了まで待機する
            if (_queueLimitTask != null)
            {
                _queueLimitTask.Task.Wait();
                _queueLimitTask = null;
            }
        }

        public Task<T> Take()
        {
            return Take(default(CancellationToken));
        }

        public Task<T> Take(CancellationToken token)
        {
            if (token.WaitHandle != null)
            {
                token.Register(() => canceled());
            }

            lock (_collection)
            {
                if (_collection.Count > 0)
                {
                    // データのキューが空でなければキューから取り出す
                    //return Task.FromResult(_collection.Dequeue()); // .NET Framework 4.0 には FromResult メソッドがない
                    return Task<T>.Factory.StartNew(() =>
                    {
                        var item = _collection.Dequeue();
                        System.Diagnostics.Trace.WriteLine(
                            string.Format("{0} データキューの数: {1}",
                            DateTime.Now.ToLongTimeString(), _collection.Count));
                        // キューへの投入待機用オブジェクトが生成されていた場合には「正常に完了」への遷移を試みる
                        if (_queueLimitTask != null)
                        {
                            _queueLimitTask.TrySetResult(true);
                        }
                        return item;
                    });
                }
                else
                {
                    // キューが空だったら、未完了状態の TaskCompletionSource から作成されるタスクを返す
                    var tcs = new TaskCompletionSource<T>();
                    _waiting.Enqueue(tcs);
                    return tcs.Task;
                }
            }
        }

        private void canceled()
        {
            lock (_collection)
            {
                // キューへの投入待機用オブジェクトが生成されていた場合には「正常に完了」への遷移を試みる
                if (_queueLimitTask != null)
                {
                    _queueLimitTask.TrySetResult(true);
                }

                // キューに入っているデータをクリア
                while (_collection.Count > 0)
                {
                    var item = _collection.Dequeue();
                    if (item as IDisposable != null)
                    {
                        (item as IDisposable).Dispose();
                    }
                }

                while (_waiting.Count > 0)
                {
                    // 未完了状態の TaskCompletionSource オブジェクトを取り出し
                    var tcs = _waiting.Dequeue();
                    // Canceld 状態への遷移を試みる
                    tcs.TrySetCanceled();
                }
            }
        }
    }
}
