﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace MultiThreadAcceptTest2.Server
{
    class ServerService : IDisposable
    {
        private int _concurrencyLimit;
        private Encoding _encoding;
        private AsyncProducerConsumerCollection<TcpClient> _queue;
        private List<Task> _tasks;
        private TcpListener _listener;
        private bool _disposed = false;
        private const int QueueLimit = 2; // キュー投入の制限数

        // スレッドで例外が発生した場合に発生するイベント
        public event ThreadExceptionEventHandler ThreadExceptionOccurred;

        public ServerService(int concurrencyLimit, Encoding encoding)
        {
            _concurrencyLimit = concurrencyLimit;
            _encoding = encoding;
            _queue = new AsyncProducerConsumerCollection<TcpClient>(QueueLimit);
            _tasks = new List<Task>();
            _listener = null;
        }

        ~ServerService()
        {
            Dispose(false);
        }

        private Action cancel;

        public void StartService(string hostAddress, int portNo)
        {
            try
            {
                var ipAddress = checkHostAddress(hostAddress);
                checkPortNo(portNo);

                var cts = new CancellationTokenSource();
                cancel = () => cts.Cancel();

                // コンシューマを ConcurrencyLimit で指定された数立ち上げる
                for (var i = 0; i < _concurrencyLimit; ++i)
                {
                    _tasks.Add(TaskConsumerAsync(cts.Token));
                }

                if (_listener == null)
                {
                    _tasks.Add(TaskProducerAsync(ipAddress, portNo, cts.Token));
                }
            }
            catch (ServerServiceException e)
            {
                raiseThreadExceptionOccurred(e);
            }
        }

        public void StopService()
        {
            if (cancel != null)
            {
                cancel();
                cancel = null;
            }

            if (_listener != null)
            {
                _listener.Stop();
                _listener = null;
            }
        }

        public void WaitTasksEnd()
        {
            Task.WaitAll(_tasks.ToArray());
        }

        private IPAddress checkHostAddress(string hostAddress)
        {
            if (string.IsNullOrEmpty(hostAddress))
            {
                throw new ServerServiceException(ServerError.IpAddressError,
                    "IP アドレスが設定されていません。");
            }

            var hostName = Dns.GetHostName();
            var addresses = Dns.GetHostAddresses(hostName);
            IPAddress ipAddress;
            if (!IPAddress.TryParse(hostAddress, out ipAddress))
            {
                throw new ServerServiceException(ServerError.IpAddressError,
                    "無効なアドレスが設定されています。");
            }
            if (!IPAddress.IsLoopback(ipAddress) && !addresses.Contains(ipAddress))
            {
                throw new ServerServiceException(ServerError.IpAddressError,
                    "設定されたアドレスは自ホストのアドレスではありません。");
            }

            return ipAddress;
        }

        private void checkPortNo(int portNo)
        {
            if (portNo < 0 || portNo > 65535)
            {
                throw new ServerServiceException(ServerError.PortOutOfRange,
                    "無効なポート番号が設定されています。");
            }
            if (portNo < 1024)
            {
                throw new ServerServiceException(ServerError.PortOutOfRange,
                    "ポート番号に予約ポートが設定されています。");
            }
        }

        private async Task TaskConsumerAsync(CancellationToken token)
        {
            try
            {
                while (!token.IsCancellationRequested)
                {
                    var client = await _queue.Take(token);
                    try
                    {
                        await doProcessAsync(client, token);
                    }
                    catch (IOException e)
                    {
                        if (e.InnerException.GetType() == typeof(SocketException) &&
                            (e.InnerException as SocketException).ErrorCode == 10054)
                        {
                            Console.WriteLine("{0} [コンシューマ] クライアントが切断していたためデータを送信できませんでした。",
                                DateTime.Now.ToLongTimeString());
                        }
                    }
                    catch (OperationCanceledException)
                    {
                        Console.WriteLine("{0} [コンシューマ] 処理中にキャンセルが要求されました。",
                            DateTime.Now.ToLongTimeString());
                    }
                    finally
                    {
                        client.Close();
                    }
                }
            }
            catch (OperationCanceledException)
            {
                Console.WriteLine("{0} [コンシューマ] 処理データの待機中にキャンセルが要求されました。",
                    DateTime.Now.ToLongTimeString());
            }
        }

        private async Task doProcessAsync(TcpClient client, CancellationToken token)
        {
            using (var stream = client.GetStream())
            {
                Console.WriteLine("{0} [S] Server ready", DateTime.Now.ToLongTimeString());
                await sendDataAsync(stream, _encoding.GetBytes("Server ready"));

                var rData = await receiveDataAsync(stream);
                Console.WriteLine("{0} [C] {1}", DateTime.Now.ToLongTimeString(), _encoding.GetString(rData));

                token.ThrowIfCancellationRequested();

                // サーバー側処理の代わりに15秒待つ
                //(並行処理の様子の観察に都合が良い)
                await Task.Delay(5000);
                token.ThrowIfCancellationRequested();

                await Task.Delay(5000);
                token.ThrowIfCancellationRequested();

                await Task.Delay(5000);
                token.ThrowIfCancellationRequested();

                Console.WriteLine("{0} [S] {1}", DateTime.Now.ToLongTimeString(), _encoding.GetString(rData));
                await sendDataAsync(stream, rData);
            }
        }

        private async Task TaskProducerAsync(IPAddress ipAddress, int portNo, CancellationToken token)
        {
            try
            {
                _listener = new TcpListener(ipAddress, portNo);
                _listener.Start();
                Console.WriteLine("{0} {1}/Port[{2}]の Listen を開始しました。",
                    DateTime.Now.ToLongTimeString(), ipAddress.ToString(), portNo);

                while (!token.IsCancellationRequested)
                {
                    var client = await _listener.AcceptTcpClientAsync();
                    Console.WriteLine("{0} クライアントが接続しました。",
                        DateTime.Now.ToLongTimeString());
                    // 接続してきたクライアントをキューにセットする
                    _queue.Add(client);

                }
            }
            catch (SocketException e)
            {
                switch (e.ErrorCode)
                {
                    case 10004:
                        Console.WriteLine("{0} Port[{1}]の Listen が強制終了されました。",
                            DateTime.Now.ToLongTimeString(), portNo);
                        break;
                    default:
                        Console.WriteLine("{0} Socket exception: {1}, errCode({2})",
                            DateTime.Now.ToLongTimeString(), e.Message, e.ErrorCode);
                        break;
                }
            }
            catch (ObjectDisposedException)
            {
                Console.WriteLine("{0} Port[{1}]の Listen が破棄されました。",
                    DateTime.Now.ToLongTimeString(), portNo);
            }
            finally
            {
                if (_listener != null)
                {
                    _listener.Stop();
                }
            }
        }

        private async Task<byte[]> receiveDataAsync(NetworkStream stream)
        {
            byte[] result;
            using (var memStream = new MemoryStream())
            {
                var rBuff = new byte[1024];
                int rSize;
                do
                {
                    // データの一部を受信する
                    rSize = await stream.ReadAsync(rBuff, 0, rBuff.Length);
                    // rSize が 0 のときにはクライアントが切断したと判断
                    if (rSize == 0)
                    {
                        Console.WriteLine("クライアントが切断しました。");
                        throw new ServerServiceException(ServerError.ConnectionClose,
                            "ストリームの読み出しの際、クライアントが切断していました。");
                    }
                    // 受信したデータを蓄積する
                    memStream.Write(rBuff, 0, rSize);
                } while (stream.DataAvailable);

                result = memStream.ToArray();
            }
            Console.WriteLine("RecieveData(hex):{0}", bytes2Hex(result));
            return result;
        }

        private async Task sendDataAsync(NetworkStream stream, byte[] message)
        {
            await stream.WriteAsync(message, 0, message.Length);
            Console.WriteLine("SendData(hex):{0}", bytes2Hex(message));
        }

        private static string bytes2Hex(byte[] bytes)
        {
            var val = new StringBuilder();
            foreach (var n in bytes)
            {
                val.Append(string.Format("{0:X2} ", n));
            }
            return val.ToString();
        }

        // ThreadExceptionOccurred イベントを発火させる
        private void raiseThreadExceptionOccurred(Exception e)
        {
            var handler = ThreadExceptionOccurred;
            if (handler != null)
            {
                handler(this, new ThreadExceptionEventArgs(e));
            }
        }

        #region IDisposable

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (_disposed) return;

            _disposed = true;

            if (disposing)
            {
                // マネージ リソースの解放処理
            }
            // アンマネージ リソースの解放処理
            StopService();
        }

        #endregion
    }
}
