﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Input;
using System.ComponentModel.DataAnnotations;
using ValidationTestRoomReservation.Models;
using ValidationTestRoomReservation.Services;
using MakCraft.ViewModels;

namespace ValidationTestRoomReservation.ViewModels
{
    public class ReserveWindowViewModel : ValidationViewModelBase
    {
        private IReserveService _service;
        private int _roomId;
        private DateTime _start;
        private DateTime _end;
        private string _subscriber;
        private string _logicalError;
        public ReserveWindowViewModel()
        {
            _service = new ReserveService();
            System.Diagnostics.Debug.WriteLine("ReserveWindowViewModel created!");
            Start = DateTime.Now;
            End = DateTime.Now;
        }
        public ReserveWindowViewModel(IReserveService service)
        {
            _service = service;
        }

        public int RoomId
        {
            get { return _roomId; }
            set
            {
                _roomId = value;
                RaisePropertyChanged(PropertyHelper.GetName(() => RoomName));
                RaisePropertyChanged(PropertyHelper.GetName(() => Reservations));
            }
        }

        public string RoomName
        {
            get { return _service.GetRooms().Where(w => w.Id == RoomId).First().Name; }
        }

        public IList<Reservation> Reservations
        {
            get { return _service.GetReservations(RoomId).OrderBy(w => w.Start).ToList(); }
        }

        public DateTime Start
        {
            get { return _start; }
            set
            {
                var propertyName = PropertyHelper.GetName(() => Start);
                base.RemoveItemValidationError(propertyName);
                _start = value;
                logiclCheck();
                base.RaisePropertyChanged(PropertyHelper.GetName(() => Start));
            }
        }

        public DateTime End
        {
            get {return _end;}
            set
            {
                var propertyName = PropertyHelper.GetName(() => End);
                base.RemoveItemValidationError(propertyName);
                _end = value;
                logiclCheck();
                base.RaisePropertyChanged(PropertyHelper.GetName(() => End));
            }
        }

        [Required(ErrorMessage = "登録者名は必須項目です。")]
        public string Subscriber
        {
            get { return _subscriber; }
            set
            {
                var propertyName = PropertyHelper.GetName(() => Subscriber);
                base.RemoveItemValidationError(propertyName);
                _subscriber = value;
                base.RaisePropertyChanged(propertyName);
            }
        }

        public string LogicalError
        {
            get { return _logicalError; }
            set
            {
                _logicalError = value;
                base.RaisePropertyChanged(PropertyHelper.GetName(() => LogicalError));
            }
        }

        private void addReservationCommandExecute()
        {
            base.ViewModelState.RemoveErrorByKey(PropertyHelper.GetName(() => LogicalError));
            base.ViewModelState.RemoveErrorByKey(PropertyHelper.GetName(() => Start));
            base.ViewModelState.RemoveErrorByKey(PropertyHelper.GetName(() => End));
            var isError = _service.ReserveRoom(RoomId, Start, End, Subscriber);
            switch (isError)
            {
                case IsReservError.StartTime:
                    base.ViewModelState.AddError(PropertyHelper.GetName(() => Start), "会議時間が競合しています。");
                    break;
                case IsReservError.EndTime:
                    base.ViewModelState.AddError(PropertyHelper.GetName(() => End), "会議時間が競合しています。");
                    break;
                case IsReservError.BothTime:
                    base.ViewModelState.AddError(PropertyHelper.GetName(() => Start), "会議時間が競合しています。");
                    base.ViewModelState.AddError(PropertyHelper.GetName(() => End), "会議時間が競合しています。");
                    break;
                case IsReservError.Repository:
                    base.ViewModelState.AddError(PropertyHelper.GetName(() => LogicalError),
                        "DB 登録でエラーが発生しました。");
                    break;
            }
            base.RaisePropertyChanged(PropertyHelper.GetName(() => Reservations));
            base.RaisePropertyChanged(PropertyHelper.GetName(() => Start));
            base.RaisePropertyChanged(PropertyHelper.GetName(() => End));
            base.RaisePropertyChanged(PropertyHelper.GetName(() => IsValid));
        }
        private bool addReservationCommandCanExecute
        {
            get { return base.ViewModelState.IsValid; }
        }
        private ICommand addReservationCommand;
        public ICommand AddReservationCommand
        {
            get
            {
                if (addReservationCommand == null)
                    addReservationCommand = new RelayCommand(
                        param => addReservationCommandExecute(),
                        param => addReservationCommandCanExecute
                    );
                return addReservationCommand;
            }
        }

        private void logiclCheck()
        {
            var errorList = new List<string>();
            if (!_service.IsReservValid(Start, End))
            {
                errorList.Add("会議開始日時は終了日時より前の日時を指定してください。");
            }
            if (!_service.IsReservValid(Start, End))
            {
                errorList.Add("会議終了日時は開始日時より後の日時を指定してください。。");
            }
            var propertyName = PropertyHelper.GetName(() => LogicalError);
            base.ViewModelState.RemoveErrorByKey(propertyName);
            if (errorList.Count > 0)
            {
                base.ViewModelState.AddError(propertyName,
                    string.Join(Environment.NewLine, errorList));
            }
        }

        ~ReserveWindowViewModel()
        {
            System.Diagnostics.Debug.WriteLine("ReserveWindowViewModel destructed!");
        }
    }
}
