﻿using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.IO;
using System.Threading.Tasks;
using System.Windows.Input;

using MakCraft.ViewModels;
using NetMatters;

using Clutch.Properties;

namespace Clutch.ViewModels
{
    class MainWindowViewModel : ValidationViewModelBase
    {
        public MainWindowViewModel()
        {
        }

        private string _result;
        public string Result
        {
            get { return _result; }
            set
            {
                _result = value;
                base.RaisePropertyChanged(() => Result);
            }
        }

        private string _fileName;
        [Required(ErrorMessageResourceName = "RequiredFileName", ErrorMessageResourceType = typeof(ErrorMessage))]
        public string FileName
        {
            get { return _fileName; }
            set
            {
                _fileName = value;
                base.RemoveItemValidationError(() => FileName);
                base.RaisePropertyChanged(() => FileName);
                Result = "";
            }
        }

        private async void searchClutchProcessExecute()
        {
            Mouse.OverrideCursor = Cursors.Wait;
            _searchClutchProcessExecuting = true;

            List<string> files;
            var messages = new List<string>();

            if (Directory.Exists(FileName))
                files = await Task.Factory.StartNew<List<string>>(() =>
                    new List<string>(Directory.GetFiles(FileName, "*", SearchOption.AllDirectories)));
            else if (File.Exists(FileName))
                files = new List<string> { FileName };
            else
            {
                var errMessage = Properties.ErrorMessage.FileNotFound;
                base.ViewModelState.AddError(PropertyHelper.GetName(() => FileName), errMessage);
                base.RaisePropertyChanged(() => FileName);
                Result = errMessage;
                Mouse.OverrideCursor = null;
                _searchClutchProcessExecuting = false;
                // コマンド実行可否評価を強制実行
                // 非同期処理にしたら、ボタンの無効化が解除されなくなったため
                CommandManager.InvalidateRequerySuggested();
                return;
            }

            var processCount = await Task.Factory.StartNew<int>(() => searchClutchFile(files, messages));

            Result = string.Format(Properties.NotifyMessage.FoundProcess, processCount) + "\r\n" +
                Properties.NotifyMessage.FoundProcessTitle + "\r\n" + string.Join("\r\n", messages.ToArray());

            Mouse.OverrideCursor = null;
            _searchClutchProcessExecuting = false;
            // コマンド実行可否評価を強制実行
            // 非同期処理にしたら、ボタンの無効化が解除されなくなったため
            CommandManager.InvalidateRequerySuggested();    
        }
        private bool _searchClutchProcessExecuting = false;
        private bool _searchClutchProcessCanExecute
        {
            get { return !_searchClutchProcessExecuting && base.IsValid; }
        }
        private ICommand _searchClutchProcessCommand;
        public ICommand SearchClutchProcessCommand
        {
            get
            {
                if (_searchClutchProcessCommand == null)
                    _searchClutchProcessCommand = new RelayCommand(
                        param => searchClutchProcessExecute(),
                        param => _searchClutchProcessCanExecute
                        );
                return _searchClutchProcessCommand;
            }
        }

        private int searchClutchFile(List<string> filePaths, List<string> resuls)
        {
            foreach (var n in filePaths)
            {
                if (!File.Exists(n))
                {
                    var errMessage = Properties.ErrorMessage.FileNotFound;
                    base.ViewModelState.AddError(PropertyHelper.GetName(() => FileName), errMessage);
                    base.RaisePropertyChanged(() => FileName);
                    Result = errMessage;
                    return 0;
                }
            }

            var processes = InUseDetection.GetProcessesUsingFiles(filePaths);
            if (processes.Count == 0)
            {
                Result = Properties.NotifyMessage.ProcessNotFound;
                return 0;
            }

            foreach (var n in processes)
            {
                resuls.Add(string.Format("{0}: {1}, {2}", n.Process.ProcessName, n.Process.Id, n.FilePath));
            }

            return processes.Count;
        }
    }
}
