﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace GetJpegInfo.Graphics
{
    public class JpegInfo : GraphicsInfoBase
    {
        public static readonly UInt32 MARKER_OFFSET = 2;
        private const UInt32 TIFF_HEADER_LENGTH = 8;

        private const UInt32 EXIF_HEADER_LENGTH = 10;
        private const UInt32 EXIF_IFD_ENTRY_LEGTH = 12;
        private FormatType _formatType = FormatType.Undefined;
        private Dictionary<SegType, List<UInt32>> _segments = new Dictionary<SegType, List<uint>>();
        private List<string> _note = new List<string>();
        private SegType _sof = 0;
        private UInt32 _exifIFDPointer = 0;
        private UInt32 _gpsIFDPointer = 0;
        private UInt32 _exifTiffHeaderOffset = 0;
        private bool _isThumbnail = false;

        public JpegInfo() { }
        public JpegInfo(byte[] source, IList<string> message, PicFormat format) : base(source, message, format) { }
        public JpegInfo(byte[] source, IList<string> message, PicFormat format, bool isThumbnail) : base(source, message, format)
        {
            _isThumbnail = isThumbnail;
        }

        protected override bool check()
        {
            UInt32 offset = 0;
            SegType segType = 0;
            try
            {
                offset = checkSoiSeg(offset); // ファイル先頭の SOI の確認
                while (offset < base.SourceLength)
                {
                    offset = checkSegment(offset, out segType);
                }
                checkFormat();
            }
            catch (NotifyException e)
            {
                base.Messages.Add(e.Message);
                return false;
            }
            catch (OutOfRangeException)
            {
                base.Messages.Add("ファイルは画像ファイルではありません。");
                return false;
            }

            makeCheckMessage();
            return true;
        }

        private UInt32 checkSegment(UInt32 offset, out SegType segType)
        {
            if (base.GetByte(offset) != 0xFF)
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }
            segType = (SegType)base.GetByte(offset + 1);
            if (!SegExplanation.ContainsKey(segType))
            {
                throw new NotifyException("JPEG 形式と思われますが、未知のセグメント タイプを検出しました。");
            }
            switch (segType)
            {
                case SegType.SOI:
                    offset = checkSoiSeg(offset);
                    break;
                case SegType.EOI:
                    offset = checkEoiSeg(offset);
                    break;
                case SegType.SOS:
                    offset = checkSosSeg(offset);
                    break;
                default:
                    offset = checkSeg(offset, segType);
                    break;
            }

            return offset;
        }

        private UInt32 checkSoiSeg(UInt32 offset)
        {
            var current = offset;
            // ファイル先頭の SOI の確認
            if (base.GetByte(current++) != 0xFF || (SegType)base.GetByte(current++) != SegType.SOI)
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }

            if (_segments.ContainsKey(SegType.SOI))
            {
                throw new NotifyException("JPEG 形式と思われますが、セグメント構成が不正です(複数の SOI セグメントを検出しました)。");
            }

            addSeg(SegType.SOI, offset);

            return current;
        }

        private UInt32 checkEoiSeg(UInt32 offset)
        {
            var current = offset;
            current += MARKER_OFFSET;
            if (base.Source.Length != current)
            {
                _note.Add(string.Format("ファイルの最後に {0} バイトのゴミが付いています。",
                    base.Source.Length - offset));
            }

            addSeg(SegType.EOI, offset);

            return base.SourceLength;
        }

        private UInt32 checkSosSeg(UInt32 offset)
        {
            addSeg(SegType.SOS, offset);
            var current = offset + MARKER_OFFSET;
            for (UInt32 i = current; i < base.Source.Length - 1; ++i)
            {
                if (base.GetByte(i) == 0xFF && base.GetByte(i + 1) != 0x00 && base.GetByte(i + 1) != 0xD0 &&
                    base.GetByte(i + 1) != 0xD1 && base.GetByte(i + 1) != 0xD2 && base.GetByte(i + 1) != 0xD3 &&
                    base.GetByte(i + 1) != 0xD4 && base.GetByte(i + 1) != 0xD5 && base.GetByte(i + 1) != 0xD6 &&
                    base.GetByte(i + 1) != 0xD7)
                {   // 0xFF のエスケープ(0xFF00)及びリスタート インターバル(0xFFD1~0xFFD7)でなければ次のマーカーの出現で SOS セグメントの終了
                    offset = i;
                    break;
                }
            }

            // SOS セグメントが終了している(次のマーカーが出現する)ことを確認
            if (offset == current)
            {
                throw new NotifyException("JPEG 形式と思われますが、データフォーマットが不正です(SOS セグメントが終了していません)。");
            }

            return offset;
        }

        private UInt32 checkSeg(UInt32 offset, SegType segType)
        {
            addSeg(segType, offset);
            if (segType == SegType.SOF0 || segType == SegType.SOF1 || segType == SegType.SOF2 || segType == SegType.SOF3 ||
                segType == SegType.SOF5 || segType == SegType.SOF6 || segType == SegType.SOF7 || segType == SegType.SOF9 ||
                segType == SegType.SOF10 || segType == SegType.SOF11 || segType == SegType.SOF13 ||
                segType == SegType.SOF14 || segType == SegType.SOF15)
            {
                if (_sof != 0)
                {
                    throw new NotifyException("JPEG 形式と思われますが、セグメント構成が不正です(複数の SOF セグメントを検出しました)。");
                }
                _sof = segType;
            }
            var current = offset + MARKER_OFFSET;
            offset += base.GetUInt16(current, Endianness.Big) + MARKER_OFFSET;

            return offset;
        }

        private void addSeg(SegType segType, UInt32 offset)
        {
            if (_segments.ContainsKey(segType))
            {
                _segments[segType].Add(offset);
            }
            else
            {
                _segments.Add(segType, new List<UInt32> { offset });
            }
        }

        private void checkFormat()
        {
            if (_segments.ContainsKey(SegType.APP0) && _segments.ContainsKey(SegType.APP1))
            {
                _formatType = FormatType.ExifOnJFIF;
            }
            else if (_segments.ContainsKey(SegType.APP0))
            {
                _formatType = FormatType.JFIF;
            }
            else if (_segments.ContainsKey(SegType.APP1))
            {
                _formatType = FormatType.Exif;
            }
            else
            {
                if (!_isThumbnail)
                {   // サムネイルでなければエラーを表示
                    _note.Add("アプリケーション セグメントが見つかりません。");
                }
            }

            if (_sof == 0)
            {
                throw new NotifyException("JPEG 形式と思われますが、データフォーマットが不正です(SOF セグメントがありません)。");
            }

            if (!_segments.ContainsKey(SegType.EOI))
            {
                throw new NotifyException("JPEG 形式と思われますが、データフォーマットが不正です(EOI セグメントがありません)。");
            }
        }

        private void makeCheckMessage()
        {
            base.Messages.Add("【JPEG ファイル構成情報】");
            switch (_formatType)
            {
                case FormatType.JFIF:
                    base.Messages.Add("JPEG(JFIF 形式) ファイルです。");
                    break;
                case FormatType.ExifOnJFIF:
                    base.Messages.Add("JPEG(JFIF 形式(Exif セグメントを含む)) ファイルです。");
                    break;
                case FormatType.Exif:
                    base.Messages.Add("JPEG(Exif 形式) ファイルです。");
                    break;
                default:
                    if (!_isThumbnail)
                    {   // サムネイル画像でなければエラーを表示
                        base.Messages.Add("不明な形式です。");
                    }
                    break;
            }
            foreach (var n in _segments.Keys)
            {
                base.Messages.Add(string.Format("{0} 個の{1}({2})セグメントを持っています。",
                    _segments[n].Count, n, SegExplanation[n]));
            }

            foreach (var n in _note)
            {
                base.Messages.Add(n);
            }
            _note.Clear();

        }

        protected override void analize()
        {
            getSofInfo();

            if (_formatType == FormatType.JFIF || _formatType == FormatType.ExifOnJFIF)
            {
                getJfifInfo();
            }
            if (_formatType == FormatType.Exif || _formatType == FormatType.ExifOnJFIF)
            {
                getExifInfo();
            }
        }

        private void getSofInfo()
        {
            var sof = new JpegSofInfo(base.Source, _segments[_sof].First());
            base.Messages.Add(string.Format("{0}【SOF 情報】", Environment.NewLine));
            base.Messages.Add(string.Format("Picture size: {0} x {1}", sof.X, sof.Y));
            switch (_sof)
            {
                case SegType.SOF0:
                    base.Messages.Add("ベースライン形式 JPEG");
                    break;
                case SegType.SOF1:
                case SegType.SOF5:
                case SegType.SOF9:
                case SegType.SOF13:
                    base.Messages.Add("シーケンシャル形式 JPEG");
                    break;
                case SegType.SOF2:
                case SegType.SOF6:
                case SegType.SOF10:
                case SegType.SOF14:
                    base.Messages.Add("プログレッシブ形式 JPEG");
                    break;
                case SegType.SOF3:
                case SegType.SOF7:
                case SegType.SOF11:
                case SegType.SOF15:
                    base.Messages.Add("プログレッシブ形式 JPEG");
                    break;
            }
        }

        private void getJfifInfo()
        {
            var jfif = new JpegJfifInfo(base.Source, _segments[SegType.APP0].First());
            base.Messages.Add(string.Format("{0}【JFIF 情報】", Environment.NewLine));
            base.Messages.Add(string.Format("Pixel Aspect Ratio(X:Y): {0}:{1}", jfif.PixelAspectX, jfif.PixelAspectY));
            base.Messages.Add(string.Format("Thumbnail size: {0} x {1}", jfif.ThumbnailX, jfif.ThumbnailY));
            base.JfifThumbnailWidth = jfif.ThumbnailX;
            base.JfifThumbnailHeight = jfif.ThumbnailY;
            if (jfif.ThumbnailX != 0 || jfif.ThumbnailY != 0)
            {
                base.HasJfifThumbnail = true;
            }
        }

        private void getExifInfo()
        {
            checkExifHeader();
            UInt32 offset = 0; 
            Endianness dataEndian;
            offset = checkTiffHeader(offset, out dataEndian);
            var ifd1Offset = ifd0Info(offset, dataEndian);
            if (_exifIFDPointer != 0)
            {
                exifIfdInfo(_exifIFDPointer, dataEndian);
            }
            if (_gpsIFDPointer != 0)
            {
                gpsIfdInfo(_gpsIFDPointer, dataEndian);
            }

            if (ifd1Offset != 0)
            {
                getThumbnailInfo(ifd1Offset, dataEndian);
            }
        }

        private void checkExifHeader()
        {
            var offset = _segments[SegType.APP1].First();
            var current = offset;
            if (base.SourceLength < offset + EXIF_HEADER_LENGTH)
            {
                throw new NotifyException("画像ファイルではありません");
            }
            current += MARKER_OFFSET;
            var length = base.GetUInt16(current, Endianness.Big) + MARKER_OFFSET;
            if (base.SourceLength < offset + length)
            {
                throw new NotifyException("画像ファイルではありません(項目長が不正です)");
            }
            current += GraphicsInfoBase.INT16_OFFSET;
            if (base.GetByte(current++) != 0x45 || base.GetByte(current++) != 0x78 || base.GetByte(current++) != 0x69 ||
                base.GetByte(current++) != 0x66 || base.GetByte(current++) != 0x00 || base.GetByte(current++) != 0x00)
            {
                throw new NotifyException("画像ファイルではありません(APP1 セグメントの識別子が不正です)");
            }

            _exifTiffHeaderOffset = current; // TIFF ヘッダーへのオフセット
        }

        // Exif 形式のシフトされた TIFF ヘッダーの開始位置で補正を行ったオフセット値を取得します。
        private UInt32 getExifTiffOffset(UInt32 offset)
        {
            return offset + _exifTiffHeaderOffset;
        }

        private UInt32 checkTiffHeader(UInt32 offset, out Endianness dataEndian)
        {
            if (offset != 0 || getExifTiffOffset(offset) + TIFF_HEADER_LENGTH > base.SourceLength)
            {
                throw new NotifyException("画像ファイルではありません。");
            }
            var current = offset;

            dataEndian = Endianness.Big;
            if (base.GetByte(getExifTiffOffset(current)) == 0x49 && base.GetByte(getExifTiffOffset(current + 1)) == 0x49)
            {
                dataEndian = Endianness.Little;
            }
            else if (base.GetByte(getExifTiffOffset(current)) != 0x4D || base.GetByte(getExifTiffOffset(current + 1)) != 0x4D)
            {
                throw new NotifyException("画像ファイルではありません。");
            }
            current += GraphicsInfoBase.INT16_OFFSET;
            if (base.GetUInt16(getExifTiffOffset(current), dataEndian) != 42u) // Tiff 識別子
            {
                throw new NotifyException("画像ファイルではありません。");
            }
            current += GraphicsInfoBase.INT16_OFFSET;
            current = base.GetUInt32(getExifTiffOffset(current), dataEndian);
            if (getExifTiffOffset(current) >= base.SourceLength)
            {
                throw new NotifyException("0th IFD へのオフセット値が不正です。");
            }

            return current;
        }

        /// <summary>
        /// IFD0 の情報を出力用のリストに追加し、IFD1 へのオフセットを取得します。
        /// </summary>
        /// <param name="offset"></param>
        /// <param name="dataEndian"></param>
        /// <returns></returns>
        private UInt32 ifd0Info(UInt32 offset, Endianness dataEndian)
        {
            JpegExifInfo ifd0Tags = new JpegExifInfo();
            offset = analizeExifIfd(offset, ifd0Tags, dataEndian);
            base.Messages.Add(string.Format("{0}【Exif(IFD0) 情報】", Environment.NewLine));
            if (ifd0Tags.ResolutionUnit != null) { base.Messages.Add(ifd0Tags.ResolutionUnit); }
            foreach (var n in ifd0Tags.TagInfos)
            {
                base.Messages.Add(n);
            }

            return offset;
        }

        private UInt32 exifIfdInfo(UInt32 offset, Endianness dataEndian)
        {
            JpegExifInfo exifTags = new JpegExifInfo();
            offset = analizeExifIfd(offset, exifTags, dataEndian);
            base.Messages.Add(string.Format("{0}【Exif IFD 情報】", Environment.NewLine));
            if (exifTags.ExifVersion != null) { base.Messages.Add(exifTags.ExifVersion); }
            foreach (var n in exifTags.TagInfos)
            {
                base.Messages.Add(n);
            }

            return offset;
        }

        private UInt32 gpsIfdInfo(UInt32 offset, Endianness dataEndian)
        {
            JpegGpsInfo gpsTags = new JpegGpsInfo();
            offset = analizeExifIfd(offset, gpsTags, dataEndian);
            base.Messages.Add(string.Format("{0}【GPS IFD 情報】", Environment.NewLine));
            if (gpsTags.GpsLatitude != null) { base.Messages.Add(gpsTags.GpsLatitude); }
            if (gpsTags.GpsLongitude != null) { base.Messages.Add(gpsTags.GpsLongitude); }
            if (gpsTags.GpsAltitude != null) { base.Messages.Add(gpsTags.GpsAltitude); }
            if (gpsTags.GpsTimeStamp != null) { base.Messages.Add(gpsTags.GpsTimeStamp); }
            foreach (var n in gpsTags.TagInfos)
            {
                base.Messages.Add(n);
            }

            return offset;
        }

        private UInt32 analizeExifIfd(UInt32 offset, ExifTags exifTags, Endianness dataEndian)
        {
            UInt16 numberEntries = base.GetUInt16(getExifTiffOffset(offset), dataEndian);
            offset += GraphicsInfoBase.INT16_OFFSET;
            for (var i = 0; i < numberEntries; ++i)
            {
                analizeExifIfdEntry(offset, exifTags, dataEndian);
                offset += EXIF_IFD_ENTRY_LEGTH;
            }

            offset = base.GetUInt16(getExifTiffOffset(offset), dataEndian); // 次の IFD へのオフセット
            return offset;
        }

        private void analizeExifIfdEntry(UInt32 offset, ExifTags tiffTags, Endianness dataEndian)
        {
            var tagNo = base.GetUInt16(getExifTiffOffset(offset), dataEndian);
            offset += GraphicsInfoBase.INT16_OFFSET;
            var tagType = base.GetUInt16(getExifTiffOffset(offset), dataEndian);
            offset += GraphicsInfoBase.INT16_OFFSET;
            var tagValAmount = base.GetUInt32(getExifTiffOffset(offset), dataEndian);
            offset += GraphicsInfoBase.INT32_OFFSET;
            var tagValueIndex = offset;

            if ((TiffTag)tagNo == TiffTag.GPSInfoIFDPointer)
            {
                _gpsIFDPointer = base.GetUInt32(getExifTiffOffset(tagValueIndex), dataEndian);
            }
            else if ((TiffTag)tagNo == TiffTag.ExifIFDPointer)
            {
                _exifIFDPointer = base.GetUInt32(getExifTiffOffset(tagValueIndex), dataEndian);
            }
            else
            {
                var tagData = new TiffTagData(base.Source, tagValAmount, getExifTiffOffset(tagValueIndex),
                    _exifTiffHeaderOffset, dataEndian, (TagType)tagType);
                tiffTags.AddTagInfo((TiffTag)tagNo, tagData);
            }
        }

        private void getThumbnailInfo(UInt32 offset, Endianness dataEndian)
        {
            JpegExifInfo ifd1Tags = new JpegExifInfo(); ;
            offset = analizeExifIfd(offset, ifd1Tags, dataEndian);
            if (offset != 0)
            {
                throw new NotifyException("JPEG 形式と思われますが、データフォーマットが不正です(Exif 形式の 1st IFD の後に IFD が存在します)。");
            }
            base.Messages.Add(string.Format("{0}【Exif サムネイル(IFD1)情報】", Environment.NewLine));
            foreach (var n in ifd1Tags.TagInfos)
            {
                base.Messages.Add(n);
            }

            if (base.SourceLength < getExifTiffOffset(ifd1Tags.JPEGInterchangeFormat) + ifd1Tags.JPEGInterchangeFormatLength)
            {
                throw new NotifyException("JPEG 形式と思われますが、データフォーマットが不正です(Exif 形式のサムネイル画像の長さが不正)。");
            }
            base.ExifThumbnail = new byte[ifd1Tags.JPEGInterchangeFormatLength];
            Array.Copy(base.Source, getExifTiffOffset(ifd1Tags.JPEGInterchangeFormat), base.ExifThumbnail, 0, ifd1Tags.JPEGInterchangeFormatLength);
            var thumbnailInfo = new JpegInfo(base.ExifThumbnail, base.Messages, PicFormat.JPEG, true);
            thumbnailInfo.ImageCheck();
            base.HasExifThumbnail = true;
        }

        private static readonly Dictionary<SegType, string> SegExplanation = new Dictionary<SegType, string> {
            { SegType.SOF0, "フレームタイプ0開始" },
            { SegType.SOF1, "フレームタイプ1開始" },
            { SegType.SOF2, "フレームタイプ2開始" },
            { SegType.SOF3, "フレームタイプ3開始" },
            { SegType.DHT, "ハフマン法テーブル定義" },
            { SegType.SOF5, "フレームタイプ5開始" },
            { SegType.SOF6, "フレームタイプ6開始" },
            { SegType.SOF7, "フレームタイプ7開始" },
            { SegType.SOF9, "フレームタイプ9開始" },
            { SegType.SOF10, "フレームタイプ10開始" },
            { SegType.SOF11, "フレームタイプ11開始" },
            { SegType.DAC, "算術式圧縮テーブル定義" },
            { SegType.SOF13, "フレームタイプ13開始" },
            { SegType.SOF14, "フレームタイプ14開始" },
            { SegType.SOF15, "フレームタイプ15開始" },
            { SegType.SOI, "イメージ開始" },
            { SegType.EOI, "イメージ終了" },
            { SegType.SOS, "スキャン開始" },
            { SegType.DQT, "量子化テーブル定義" },
            { SegType.DNL, "行数定義" },
            { SegType.DRI, "リスタート間隔定義" },
            { SegType.DHP, "階層行数定義" },
            { SegType.EXP, "標準イメージ拡張参照" },
            { SegType.APP0, "タイプ0のアプリケーション" },
            { SegType.APP1, "タイプ1のアプリケーション" },
            { SegType.APP2, "タイプ2のアプリケーション" },
            { SegType.APP3, "タイプ3のアプリケーション" },
            { SegType.APP4, "タイプ4のアプリケーション" },
            { SegType.APP5, "タイプ5のアプリケーション" },
            { SegType.APP6, "タイプ6のアプリケーション" },
            { SegType.APP7, "タイプ7のアプリケーション" },
            { SegType.APP8, "タイプ8のアプリケーション" },
            { SegType.APP9, "タイプ9のアプリケーション" },
            { SegType.APP10, "タイプ10のアプリケーション" },
            { SegType.APP11, "タイプ11のアプリケーション" },
            { SegType.APP12, "タイプ12のアプリケーション" },
            { SegType.APP13, "タイプ13のアプリケーション" },
            { SegType.APP14, "タイプ14のアプリケーション" },
            { SegType.APP15, "タイプ15のアプリケーション" },
            { SegType.COM, "コメント" },
        };

        private enum SegType : byte
        {
            SOF0 = 0xC0,
            SOF1 = 0xC1,
            SOF2 = 0xC2,
            SOF3 = 0xC3,
            DHT = 0xC4,
            SOF5 = 0xC5,
            SOF6 = 0xC6,
            SOF7 = 0xC7,
            SOF9 = 0xC9,
            SOF10 = 0xCA,
            SOF11 = 0xCB,
            DAC = 0xCC,
            SOF13 = 0xCD,
            SOF14 = 0xCE,
            SOF15 = 0xCF,
            SOI = 0xD8,
            EOI = 0xD9,
            SOS = 0xDA,
            DQT = 0xDB,
            DNL = 0xDC,
            DRI = 0xDD,
            DHP = 0xDE,
            EXP = 0xDF,
            APP0 = 0xE0,
            APP1 = 0xE1,
            APP2 = 0xE2,
            APP3 = 0xE3,
            APP4 = 0xE4,
            APP5 = 0xE5,
            APP6 = 0xE6,
            APP7 = 0xE7,
            APP8 = 0xE8,
            APP9 = 0xE9,
            APP10 = 0xEA,
            APP11 = 0xEB,
            APP12 = 0xEC,
            APP13 = 0xED,
            APP14 = 0xEE,
            APP15 = 0xEF,
            COM = 0xFE,
        }

        private enum FormatType
        {
            JFIF,
            ExifOnJFIF,
            Exif,
            Undefined
        }
    }
}
