﻿using System;

namespace GetJpegInfo.Graphics
{
    /// <summary>
    /// JPEG JFIF 形式の APP0 セグメントの情報を取得・保持します。
    /// </summary>
    public class JpegJfifInfo
    {
        private const byte IDENTIFIES_LENGTH = 5;

        public JpegJfifInfo() { }
        public JpegJfifInfo(byte[] source, UInt32 offset)
        {
            SetData(source, offset);
        }

        public void SetData(byte[] source, UInt32 offset)
        {
            var current = offset;
            current += JpegInfo.MARKER_OFFSET;
            var length = GraphicsInfoBase.GetUInt16(source, current, Endianness.Big) + JpegInfo.MARKER_OFFSET;
            if (source.Length < offset + length)
            {
                throw new NotifyException("画像ファイルではありません(項目長が不正です)");
            }
            current += GraphicsInfoBase.INT16_OFFSET;
            if (GraphicsInfoBase.GetStringUntilNull(source, current, IDENTIFIES_LENGTH) != "JFIF")
            {
                throw new NotifyException("画像ファイルではありません(APP0 セグメントの識別子が不正です)");
            }
            current += IDENTIFIES_LENGTH;
            VerMajor = GraphicsInfoBase.GetByte(source, current++);
            VerMinor = GraphicsInfoBase.GetByte(source, current++);
            Unit = (JfifUnit)GraphicsInfoBase.GetByte(source, current++);
            PixelAspectX = GraphicsInfoBase.GetUInt16(source, current, Endianness.Big);
            current += GraphicsInfoBase.INT16_OFFSET;
            PixelAspectY = GraphicsInfoBase.GetUInt16(source, current, Endianness.Big);
            current += GraphicsInfoBase.INT16_OFFSET;
            ThumbnailX = GraphicsInfoBase.GetByte(source, current++);
            ThumbnailY = GraphicsInfoBase.GetByte(source, current++);
            Thumbnail = null;
            if (current < offset + length)
            {
                Array.Copy(source, current, Thumbnail, 0, (offset + length) - current);
            }
        }

        public byte VerMajor { get; private set; }
        public byte VerMinor { get; private set; }

        public JfifUnit Unit { get; private set; }

        public ushort PixelAspectX { get; private set; }
        public ushort PixelAspectY { get; private set; }

        public bool HaveThumbnail
        {
            get { return ThumbnailX != 0 || ThumbnailY != 0; }
        }
        public byte ThumbnailX { get; private set; }
        public byte ThumbnailY { get; private set; }
        public byte[] Thumbnail { get; private set; }
    }

    public enum JfifUnit
    {
        NoUnits = 0,
        inch,
        cm,
    }
}
