﻿using System;
using System.Collections.Generic;

namespace GetJpegInfo.Graphics
{
    /// <summary>
    /// JPEG SOF セグメントの情報を取得・保持します。
    /// </summary>
    class JpegSofInfo
    {
        private const byte Nf_OFFSET = 8;
        private const byte SET_LENGTH = 3;

        public JpegSofInfo() { }
        public JpegSofInfo(byte[] source, UInt32 offset)
        {
            SetData(source, offset);
        }

        public void SetData(byte[] source, UInt32 offset)
        {
            var current = offset;
            current += JpegInfo.MARKER_OFFSET;
            var length = GraphicsInfoBase.GetUInt16(source, current, Endianness.Big) + JpegInfo.MARKER_OFFSET;
            if (source.Length < offset + length || length < Nf_OFFSET)
            {
                throw new NotifyException("画像ファイルではありません(項目長が不正です)");
            }
            current += GraphicsInfoBase.INT16_OFFSET;
            P = GraphicsInfoBase.GetByte(source, current++);
            Y = GraphicsInfoBase.GetUInt16(source, current, Endianness.Big);
            current += GraphicsInfoBase.INT16_OFFSET;
            X = GraphicsInfoBase.GetUInt16(source, current, Endianness.Big);
            current += GraphicsInfoBase.INT16_OFFSET;
            Nf = GraphicsInfoBase.GetByte(source, current++);
            if (length < Nf_OFFSET + Nf * SET_LENGTH)
            {
                throw new NotifyException("画像ファイルではありません(項目長が不正です)");
            }
            for (var i = 0; i < Nf; ++i)
            {
                _ci.Add(GraphicsInfoBase.GetByte(source, current++));
                _hv.Add(GraphicsInfoBase.GetByte(source, current++));
                _tqi.Add(GraphicsInfoBase.GetByte(source, current++));
            }
        }

        /// <summary>
        /// サンプルの精度
        /// </summary>
        public byte P { get; private set; }

        /// <summary>
        /// イメージの行数(高さ)
        /// </summary>
        public UInt16 Y { get; private set; }
        /// <summary>
        /// イメージの列数(横幅)
        /// </summary>
        public UInt16 X { get; private set; }

        /// <summary>
        /// 構成要素数
        /// </summary>
        public byte Nf { get; private set; }

        private List<byte> _ci = new List<byte>();
        /// <summary>
        /// 構成要素識別子
        /// </summary>
        public IReadOnlyList<byte> Ci
        {
            get { return _ci; }
        }

        private List<byte> _hv = new List<byte>();
        /// <summary>
        /// サンプル化要因
        /// </summary>
        public IReadOnlyList<byte> HV
        {
            get { return _hv; }
        }

        private List<byte> _tqi = new List<byte>();
        public IReadOnlyList<byte> Tqi
        {
            get { return _tqi; }
        }
    }
}
