﻿using System;
using System.Collections.Generic;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Text;

namespace GetJpegInfo.Graphics
{
    public class PngInfo : GraphicsInfoBase
    {
        private const UInt32 IHDR_DATA_LENGTH = 13u;
        private bool isFirstChunk = true;
        private Dictionary<ChunkType, List<UInt32>> _chunks = new Dictionary<ChunkType, List<uint>>();
        private List<string> _note = new List<string>();

        public PngInfo() { }
        public PngInfo(byte[] source, IList<string> message, PicFormat format) : base(source, message, format) { }

        protected override bool check()
        {
            UInt32 offset = 0;
            ChunkType chunkType = 0;
            try
            {
                offset = checkFileHeader(offset);
                while (chunkType != ChunkType.IEND && offset < base.SourceLength)
                {
                    offset = checkChunks(offset, out chunkType);
                }
                checkIendChunk(offset);
            }
            catch (NotifyException e)
            {
                base.Messages.Add(e.Message);
                return false;
            }
            catch (OutOfRangeException)
            {
                base.Messages.Add("ファイルは画像ファイルではありません。");
                return false;
            }

            makeCheckMessage();
            return true;
        }

        protected override void analize()
        {
            base.Messages.Add(string.Format("{0}【IHDR 情報】", Environment.NewLine));
            if (_chunks[ChunkType.IHDR].Count != 1)
            {
                throw new NotifyException("PNG 形式と思われますが、データフォーマットが不正です(IHDRチャンクの数が0あるいは複数あります)。");
            }
            analize_IHDR(_chunks[ChunkType.IHDR].First());
            if (_chunks.ContainsKey(ChunkType.tEXt) && _chunks[ChunkType.tEXt].Count > 0)
            {
                base.Messages.Add(string.Format("{0}【tEXt 情報】", Environment.NewLine));
                foreach (var n in _chunks[ChunkType.tEXt])
                {
                    analize_tEXt(n);
                }
            }
            if (_chunks.ContainsKey(ChunkType.iTXt) && _chunks[ChunkType.iTXt].Count > 0)
            {
                base.Messages.Add(string.Format("{0}【iTXt 情報】", Environment.NewLine));
                foreach (var n in _chunks[ChunkType.iTXt])
                {
                    analize_iTXt(n);
                }
            }
            if (_chunks.ContainsKey(ChunkType.tIME) && _chunks[ChunkType.tIME].Count > 0)
            {
                base.Messages.Add(string.Format("{0}【tIME 情報】", Environment.NewLine));
                if (_chunks[ChunkType.tIME].Count != 1)
                {
                    throw new NotifyException("PNG 形式と思われますが、データフォーマットが不正です(tIMEチャンクが複数あります)。");
                }
                analize_tIME(_chunks[ChunkType.tIME].First());
            }
            
        }

        private UInt32 checkFileHeader(UInt32 offset)
        {
            if (base.GetByte(offset++) != 0x89 || base.GetByte(offset++) != 0x50 || base.GetByte(offset++) != 0x4E ||
                base.GetByte(offset++) != 0x47 || base.GetByte(offset++) != 0x0D || base.GetByte(offset++) != 0x0A ||
                base.GetByte(offset++) != 0x1A || base.GetByte(offset++) != 0x0A)
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }

            return offset;
        }

        private UInt32 checkChunks(UInt32 offset, out ChunkType chunkType)
        {
            var current = offset;
            if (offset + GraphicsInfoBase.INT32_OFFSET >= base.SourceLength)
            {
                throw new NotifyException("PNG 形式と思われますが、データフォーマットが不正です(IEND チャンクがありません)。");
            }
            var length = base.GetUInt32(current, Endianness.Big);
            current += GraphicsInfoBase.INT32_OFFSET;
            chunkType = (ChunkType)base.GetUInt32(current, Endianness.Big);

            // 最初のチャンクが IHDR であることを確認
            if (isFirstChunk)
            {
                if (chunkType != ChunkType.IHDR)
                {
                    throw new NotifyException("PNG 形式と思われますが、データフォーマットが不正です(最初のチャンクが IHDR ではありません)。");
                }
                isFirstChunk = false;
            }
            addChunk(chunkType, offset);
            current += 2 * GraphicsInfoBase.INT32_OFFSET + length;

            return current;
        }

        private void addChunk(ChunkType chunkType, UInt32 offset)
        {
            if (_chunks.ContainsKey(chunkType))
            {
                _chunks[chunkType].Add(offset);
            }
            else
            {
                _chunks.Add(chunkType, new List<uint> { offset });
            }
        }

        private void checkIendChunk(UInt32 offset)
        {
            if (!_chunks.ContainsKey(ChunkType.IEND))
            {
                throw new NotifyException("PNG 形式と思われますが、データフォーマットが不正です(IEND チャンクがありません)。");
            }
            if (base.Source.Length != offset)
            {
                _note.Add(string.Format("ファイルの最後に {0} バイトのゴミが付いています。",
                    base.Source.Length - offset));
            }
        }

        private void makeCheckMessage()
        {
            base.Messages.Add("【PNG ファイル構成情報】");

            var unknownas = 0;
            foreach (var n in _chunks.Keys)
            {
                var message = string.Format("{0}: {1}", n, _chunks[n].First());
                System.Diagnostics.Trace.WriteLine(message);
                if (ChunkExplanation.ContainsKey(n))
                {
                    base.Messages.Add(string.Format("{0} 個の {1}({2})チャンクを持っています。",
                        _chunks[n].Count, n, ChunkExplanation[n]));
                }
                else
                {
                    System.Diagnostics.Trace.WriteLine(n);
                    ++unknownas;
                }
            }
            if (unknownas != 0)
            {
                base.Messages.Add(string.Format("{0} 種類のその他のチャンクを持っています。",
                    unknownas));
            }

            foreach (var n in _note)
            {
                base.Messages.Add(n);
            }
            _note.Clear();
        }

        private void analize_IHDR(UInt32 offset)
        {
            var length = base.GetUInt32(offset, Endianness.Big);
            if (length != IHDR_DATA_LENGTH)
            {
                throw new NotifyException("PNG 形式と思われますが、データフォーマットが不正です(IHDRチャンクの Length データ)。");
            }
            offset += 2 * GraphicsInfoBase.INT32_OFFSET;
            var width = base.GetUInt32(offset, Endianness.Big);
            offset += GraphicsInfoBase.INT32_OFFSET;
            var height = base.GetUInt32(offset, Endianness.Big);
            base.Messages.Add(string.Format("Picture size: {0} x {1}", width, height));
            offset += GraphicsInfoBase.INT32_OFFSET;
            var bitDepth = base.GetByte(offset++);
            var colorType = base.GetByte(offset++);
            base.Messages.Add(string.Format("Color type: {0}", getColorTypeMessage(colorType)));
        }

        private void analize_iTXt(UInt32 offset)
        {
            var current = offset;
            offset += 2 * GraphicsInfoBase.INT32_OFFSET; // データ長の基準位置となるように補正
            var length = base.GetUInt32(current, Endianness.Big);
            current += 2 * GraphicsInfoBase.INT32_OFFSET;
            string keyWord;
            current += base.GetStringUntilNull(current, length - (current - offset), out keyWord);
            current++; // Null separator 分進める
            var compressFlg = base.GetByte(current++);
            var compressMethod = base.GetByte(current++);
            string lang;
            current += base.GetStringUntilNull(current, length - (current - offset), out lang);
            current++; // Null separator 分進める
            string transKeyword;
            current += base.GetStringUntilNull(current, length - (current - offset), out transKeyword);
            current++; // Null separator 分進める
            string text;
            if (compressFlg == 0)
            {
                current += base.GetStringUntilNull(current, length - (current - offset), out text);
            }
            else if (compressFlg == 1)
            {
                text = decompress(current, length - (current - offset));
            }
            else
            {
                throw new NotifyException("iTXt チャンクの圧縮フラグの値が未知のものです。");
            }
            base.Messages.Add(string.Format("Keyword: \"{0}\", Text: \"{1}\"", keyWord, text));
        }

        private string decompress(UInt32 offset, UInt32 length)
        {
            return decompress(base.GetBytes(offset, length));
        }
        private string decompress(byte[] target)
        {
            string result = null;
            var tempFile = Path.GetTempFileName();
            using (var inSt = new MemoryStream())
            using (var outSt = new FileStream(tempFile, FileMode.Open))
            using (var dSt = new DeflateStream(inSt, CompressionMode.Decompress))
            {
                inSt.Write(target, 0, target.Length);
                inSt.Position = 0;
                dSt.CopyTo(outSt);
            }

            using (var inSt = new FileStream(tempFile, FileMode.Open))
            using (var outSt = new MemoryStream())
            {
                inSt.CopyTo(outSt);
                result = Encoding.UTF8.GetString(outSt.ToArray());
            }

            File.Delete(tempFile);

            return result;
        }

        private void analize_tEXt(UInt32 offset)
        {
            var current = offset;
            offset += 2 * GraphicsInfoBase.INT32_OFFSET; // データ長の基準位置となるように補正
            var length = base.GetUInt32(current, Endianness.Big);
            current += 2 * GraphicsInfoBase.INT32_OFFSET;
            string keyWord;
            current += base.GetStringUntilNull(current, length - (current - offset), out keyWord);
            current++; // Null separator 分進める
            string text;
            current += base.GetStringUntilNull(current, length - (current - offset), out text);
            base.Messages.Add(string.Format("Keyword: \"{0}\", Text: \"{1}\"", keyWord, text));
        }

        private void analize_tIME(UInt32 offset)
        {
            var length = base.GetUInt32(offset, Endianness.Big);
            if (length != 7)
            {
                throw new NotifyException("tIME チャンクのデータ長が不正です。");
            }
            offset += 2 * GraphicsInfoBase.INT32_OFFSET;
            var year = base.GetInt16(offset, Endianness.Big);
            offset += GraphicsInfoBase.INT16_OFFSET;
            var month = base.GetByte(offset++);
            var day = base.GetByte(offset++);
            var hour = base.GetByte(offset++);
            var minute = base.GetByte(offset++);
            var second = base.GetByte(offset);
            base.Messages.Add(string.Format("Last modified: {0}/{1}/{2} {3}:{4}:{5}", year, month, day, hour, minute, second));
        }

        private string getColorTypeMessage(byte colorType)
        {
            string result = null;
            switch (colorType)
            {
                case 0:
                    result = "グレースケール";
                    break;
                case 2:
                    result = "トゥルーカラー";
                    break;
                case 3:
                    result = "インデックスカラー(パレットあり)";
                    break;
                case 4:
                    result = "グレースケール(アルファチャネルあり)";
                    break;
                case 6:
                    result = "トゥルーカラー(アルファチャネルあり)";
                    break;
            }

            return result;
        }

        private static readonly Dictionary<ChunkType, string> ChunkExplanation = new Dictionary<ChunkType, string> {
            { ChunkType.IDAT, "Image data"},
            { ChunkType.IEND, "Image trailer"},
            { ChunkType.IHDR, "Image header"},
            { ChunkType.PLTE, "Palette"},
            { ChunkType.bKGD, "Background color"},
            { ChunkType.cHRM, "Primary chromaticities"},
            { ChunkType.gAMA, "Image gamma"},
            { ChunkType.hIST, "Palette histogram"},
            { ChunkType.iCCP, "Embedded ICC profile"},
            { ChunkType.iTXt, "International textual data"},
            { ChunkType.pHYs, "Physical pixel dimensions"},
            { ChunkType.sBIT, "Significant bits"},
            { ChunkType.sPLT, "Suggested palette"},
            { ChunkType.sRGB, "Standard RGB color space"},
            { ChunkType.tEXt, "Textual data"},
            { ChunkType.tIME, "Image last-modification time"},
            { ChunkType.tRNS, "Transparency"},
            { ChunkType.zTXt, "Compressed textual data"},
        };

        public enum ChunkType : uint
        {
            IDAT = 1229209940,
            IEND = 1229278788,
            IHDR = 1229472850,
            PLTE = 1347179589,
            bKGD = 1649100612,
            cHRM = 1665684045,
            gAMA = 1732332865,
            hIST = 1749635924,
            iCCP = 1766015824,
            iTXt = 1767135348,
            pHYs = 1883789683,
            sBIT = 1933723988,
            sPLT = 1934642260,
            sRGB = 1934772034,
            tEXt = 1950701684,
            tIME = 1950960965,
            tRNS = 1951551059,
            zTXt = 2052348020,
        }
    }
}
