﻿using System;
using System.Linq;

namespace GetJpegInfo.Graphics
{
    public class TiffTagData
    {
        private const UInt32 DATA_FIELD_LEN = 4; // IDF エントリのデータフィールド長
        private const UInt32 BYTE_LEN = 1;
        private const UInt32 INT16_LEN = 2;
        private const UInt32 INT32_LEN = 4;
        private const UInt32 INT64_LEN = 8;

        public TiffTagData(byte[] source, UInt32 tagValAmount, UInt32 offset, Endianness sourceEndian, TagType tagType) :
            this(source, tagValAmount, offset, 0, sourceEndian, tagType) { }
        public TiffTagData(byte[] source, UInt32 tagValAmount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian, TagType tagType)
        {
            _TagType = tagType;
            switch (_TagType)
            {
                case TagType.BYTE:
                    getByte(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.ASCII:
                    getAscii(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.SHORT:
                    getShort(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.LONG:
                    getLong(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.RATIONAL:
                    getRational(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.SIGNED_BYTE:
                    getSByte(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.SIGNED_SHORT:
                    getSShort(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.SIGNED_LONG:
                    getSLong(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.SIGNED_RATIONAL:
                    getSRational(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.SINGLE_FLOAT:
                    getSingle(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case TagType.DOUBLE_FLOAT:
                    getDouble(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
                case Graphics.TagType.UNDEFINED:
                    getByte(source, tagValAmount, offset, tiffHeaderOffset, sourceEndian);
                    break;
            }
        }

        private TagType _TagType;
        public TagType TagType
        {
            get { return _TagType; }
        }

        private byte[] _byteData;
        public byte[] ByteData
        {
            get
            {
                if (_TagType != TagType.BYTE && _TagType != TagType.UNDEFINED)
                {
                    throw new NotifyException("TagType.BYTE のデータは保持していません。");
                }
                return _byteData;
            }
        }

        private string _asciiData;
        public string AsciiData
        {
            get
            {
                if (_TagType != TagType.ASCII)
                {
                    throw new NotifyException("TagType.ASCII のデータは保持していません。");
                }
                return _asciiData;
            }
        }

        private UInt16[] _shortData;
        public UInt16[] ShortData
        {
            get
            {
                if (_TagType != TagType.SHORT)
                {
                    throw new NotifyException("TagType.SHORT のデータは保持していません。");
                }
                return _shortData;
            }
        }

        private UInt32[] _longData;
        public UInt32[] LongData
        {
            get
            {
                if (_TagType != TagType.LONG)
                {
                    throw new NotifyException("TagType.LONG のデータは保持していません。");
                }
                return _longData;
            }
        }

        private Rational[] _rationalData;
        public Rational[] RationalData
        {
            get
            {
                if (_TagType != TagType.RATIONAL)
                {
                    throw new NotifyException("TagType.RATIONAL のデータは保持していません。");
                }
                return _rationalData;
            }
        }

        private SByte[] _sByteData;
        public SByte[] SByteData
        {
            get
            {
                if (_TagType != TagType.SIGNED_BYTE)
                {
                    throw new NotifyException("TagType.SIGNED_BYTE のデータは保持していません。");
                }
                return _sByteData;
            }
        }

        private Int16[] _sShortData;
        public Int16[] SShortData
        {
            get
            {
                if (_TagType != TagType.SIGNED_SHORT)
                {
                    throw new NotifyException("TagType.SIGNED_SHORT のデータは保持していません。");
                }
                return _sShortData;
            }
        }

        private Int32[] _sLongData;
        public Int32[] SLongData
        {
            get
            {
                if (_TagType != TagType.SIGNED_LONG)
                {
                    throw new NotifyException("TagType.SIGNED_LONG のデータは保持していません。");
                }
                return _sLongData;
            }
        }

        private SRational[] _sRationalData;
        public SRational[] SRationalData
        {
            get
            {
                if (_TagType != TagType.SIGNED_RATIONAL)
                {
                    throw new NotifyException("TagType.SIGNED_RATIONAL のデータは保持していません。");
                }
                return _sRationalData;
            }
        }

        private float[] _singleData;
        public float[] SingleData
        {
            get
            {
                if (_TagType != TagType.SINGLE_FLOAT)
                {
                    throw new NotifyException("TagType.SINGLE_FLOAT のデータは保持していません。");
                }
                return _singleData;
            }
        }

        private double[] _doubleData;
        public double[] DoubleData
        {
            get
            {
                if (_TagType != TagType.DOUBLE_FLOAT)
                {
                    throw new NotifyException("TagType.DOUBLE_FLOAT のデータは保持していません。");
                }
                return _doubleData;
            }
        }

        public override string ToString()
        {
            string result = null;
            switch (_TagType)
            {
                case TagType.BYTE:
                    result = string.Join(", ", ByteData);
                    break;
                case TagType.ASCII:
                    result = _asciiData;
                    break;
                case TagType.SHORT:
                    result = string.Join(", ", ShortData);
                    break;
                case TagType.LONG:
                    result = string.Join(", ", LongData);
                    break;
                case TagType.RATIONAL:
                    result = string.Join(", ", RationalData.ToList());
                    break;
                case TagType.SIGNED_BYTE:
                    result = string.Join(", ", SByteData);
                    break;
                case TagType.SIGNED_SHORT:
                    result = string.Join(", ", SShortData);
                    break;
                case TagType.SIGNED_LONG:
                    result = string.Join(", ", SLongData);
                    break;
                case TagType.SIGNED_RATIONAL:
                    result = string.Join(", ", SRationalData.ToList());
                    break;
                case TagType.SINGLE_FLOAT:
                    result = string.Join(", ", SingleData);
                    break;
                case TagType.DOUBLE_FLOAT:
                    result = string.Join(", ", DoubleData);
                    break;
            }

            return result;
        }

        private void getByte(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _byteData = new byte[amount];
            if (amount > (DATA_FIELD_LEN / BYTE_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _byteData[i] = GraphicsInfoBase.GetByte(source, offset++);
            }
        }

        private void getSByte(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _sByteData = new sbyte[amount];
            if (amount > (DATA_FIELD_LEN / BYTE_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _sByteData[i] = GraphicsInfoBase.GetSByte(source, offset++);
            }
        }

        private void getAscii(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            if (amount > (DATA_FIELD_LEN / BYTE_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            _asciiData = GraphicsInfoBase.GetStringUntilNull(source, offset, amount);
        }

        private void getShort(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _shortData = new UInt16[amount];
            if (amount > (DATA_FIELD_LEN / INT16_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _shortData[i] = GraphicsInfoBase.GetUInt16(source, offset, sourceEndian);
                offset += INT16_LEN;
            }
        }

        private void getSShort(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _sShortData = new Int16[amount];
            if (amount > (DATA_FIELD_LEN / INT16_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _sShortData[i] = GraphicsInfoBase.GetInt16(source, offset, sourceEndian);
                offset += INT16_LEN;
            }
        }

        private void getLong(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _longData = new UInt32[amount];
            if (amount > (DATA_FIELD_LEN / INT32_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _longData[i] = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian);
                offset += INT32_LEN;
            }
        }

        private void getSLong(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _sLongData = new Int32[amount];
            if (amount > (DATA_FIELD_LEN / INT32_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _sLongData[i] = GraphicsInfoBase.GetInt32(source, offset, sourceEndian);
                offset += INT32_LEN;
            }
        }

        private void getSingle(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _singleData = new Single[amount];
            if (amount > (DATA_FIELD_LEN / INT32_LEN))
            {   // データはデータ部のオフセット値が示す位置にある
                offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            }
            for (var i = 0; i < amount; ++i)
            {
                _singleData[i] = GraphicsInfoBase.GetSingle(source, offset, sourceEndian);
                offset += INT32_LEN;
            }
        }

        private void getDouble(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _doubleData = new double[amount];
            // データはデータ部のオフセット値が示す位置にある
            offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            for (var i = 0; i < amount; ++i)
            {
                _doubleData[i] = GraphicsInfoBase.GetDouble(source, offset, sourceEndian);
                offset += INT64_LEN;
            }
        }

        private void getRational(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _rationalData = new Rational[amount];
            // データはデータ部のオフセット値が示す位置にある
            offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            for (var i = 0; i < amount; ++i)
            {
                _rationalData[i] = new Rational();
                _rationalData[i].Numerator = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian);
                offset += INT32_LEN;
                _rationalData[i].Denominator = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian);
                offset += INT32_LEN;
            }
        }

        private void getSRational(byte[] source, UInt32 amount, UInt32 offset, UInt32 tiffHeaderOffset, Endianness sourceEndian)
        {
            _sRationalData = new SRational[amount];
            // データはデータ部のオフセット値が示す位置にある
            offset = GraphicsInfoBase.GetUInt32(source, offset, sourceEndian) + tiffHeaderOffset;
            for (var i = 0; i < amount; ++i)
            {
                _sRationalData[i] = new SRational();
                _sRationalData[i].Numerator = GraphicsInfoBase.GetInt32(source, offset, sourceEndian);
                offset += INT32_LEN;
                _sRationalData[i].Denominator = GraphicsInfoBase.GetInt32(source, offset, sourceEndian);
                offset += INT32_LEN;
            }
        }
    }

    public class Rational
    {
        /// <summary>
        /// 分子
        /// </summary>
        public UInt32 Numerator { get; set; }
        /// <summary>
        /// 分母
        /// </summary>
        public UInt32 Denominator { get; set; }

        public override string ToString()
        {
            return string.Format("{0} / {1}", Numerator, Denominator);
        }
    }

    public class SRational
    {
        /// <summary>
        /// 分子
        /// </summary>
        public Int32 Numerator { get; set; }
        /// <summary>
        /// 分母
        /// </summary>
        public Int32 Denominator { get; set; }

        public override string ToString()
        {
            return string.Format("{0} / {1}", Numerator, Denominator);
        }
    }
}
