﻿using System;
using System.Collections.Generic;

namespace GetJpegInfo.Graphics
{
    /// <summary>
    /// GIF ApplicationExtension の情報を取得・保持します。
    /// </summary>
    class GifAppliExtInfo
    {
        private const UInt32 EXTENSION_INTRO_BLKLEN_LENGTH = 3;
        private const UInt32 APPL_IDENTIFIER_LENGTH = 8;
        private const UInt32 APPL_AUTHENTICATION_CODE_LENGTH = 3;
        private const UInt32 ANIMATED_GIF_REPETITION_OFFSET = 1;
        private const UInt32 ANIMATED_GIF_APPL_DATA_LENGTH = 3;

        public GifAppliExtInfo(byte[] source, UInt32 offset)
        {
            setData(source, offset);
        }

        public string ApplicationIdentifier { get; private set; }
        
        public byte[] ApplAuthenticationCode { get; private set; }
        
        private List<byte[]> _applicationData = new List<byte[]>();
        public IReadOnlyList<byte[]> ApplicationData
        {
            get { return _applicationData; }
        }

        public bool IsAnimatedGIF
        {
            get
            {
                if (ApplicationIdentifier != "NETSCAPE")
                {
                    return false;
                }
                UInt32 offset = 0;
                if (GraphicsInfoBase.GetByte(ApplAuthenticationCode, offset++) != 0x32 ||
                    GraphicsInfoBase.GetByte(ApplAuthenticationCode, offset++) != 0x2E ||
                    GraphicsInfoBase.GetByte(ApplAuthenticationCode, offset++) != 0x30)
                {
                    return false;
                }
                if (ApplicationData.Count != 1 || ApplicationData[0].Length != ANIMATED_GIF_APPL_DATA_LENGTH)
                {
                    return false;
                }
                var data = ApplicationData[0];
                _animationRepetitionTimes = GraphicsInfoBase.GetUInt16(data, ANIMATED_GIF_REPETITION_OFFSET, Endianness.Little);
                return true;
            }
        }

        private UInt16 _animationRepetitionTimes;
        public UInt16 AnimationRepetitionTimes
        {
            get
            {
                if (IsAnimatedGIF)
                {
                    return _animationRepetitionTimes;
                }
                else
                {
                    return 0;
                }
            }
        }

        private void setData(byte[] source, UInt32 offset)
        {
            offset += EXTENSION_INTRO_BLKLEN_LENGTH;
            ApplicationIdentifier = GraphicsInfoBase.GetStringUntilNull(source, offset, APPL_IDENTIFIER_LENGTH);
            offset += APPL_IDENTIFIER_LENGTH;
            ApplAuthenticationCode = new byte[APPL_AUTHENTICATION_CODE_LENGTH];
            Array.Copy(source, offset, ApplAuthenticationCode, 0, APPL_AUTHENTICATION_CODE_LENGTH);
            offset += APPL_AUTHENTICATION_CODE_LENGTH;
            byte length = 0;
            while (GraphicsInfoBase.GetByte(source, offset) != 0x00)
            {
                length = GraphicsInfoBase.GetByte(source, offset++);
                var item = new byte[length];
                Array.Copy(source, offset, item, 0, length);
                _applicationData.Add(item);
                offset += length;
            }
        }
    }
}
