﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace GetJpegInfo.Graphics
{
    public class GifInfo : GraphicsInfoBase
    {
        private const UInt32 HEADER_VERSION_LENGTH = 3;
        private const UInt32 LOGICAL_SCREEN_DESCRIPTOR_LENGTH = 7;
        private const UInt32 IMAGE_DESCRIPTOR_LENGTH = 10;
        private const UInt32 EXTENSION_INTRO_LABEL_LENGTH = 2;
        private const UInt32 LSD_PACKED_FIELDS_OFFSET = 4; // 論理画面記述子の色構造パラメーターへのオフセット
        private const UInt32 ID_PACKED_FIELDS_OFFSET = 9; // イメージ記述子の色構造パラメーターへのオフセット
        private const byte GRAPHIC_CTRL_EX_BLOCK_LENGTH = 4;
        private Dictionary<BlockType, List<UInt32>> _blocks = new Dictionary<BlockType, List<uint>>();
        private List<string> _note = new List<string>();
        private bool _hasGlobalColorTable = false;
        private byte _sizeOfGlobalColorTable = 0;

        public GifInfo() { }
        public GifInfo(byte[] source, IList<string> message, PicFormat format) : base(source, message, format) { }

        public VersionType Version { get; private set; } 

        protected override bool check()
        {
            UInt32 offset = 0;
            try
            {
                offset = checkHeader(offset);
                offset = checkLogicalScreenDescriptor(offset);
                if (_hasGlobalColorTable == true)
                {
                    _blocks.Add(BlockType.GlobalColorTable, new List<UInt32> { offset });
                    offset += (UInt32)Math.Pow(2, _sizeOfGlobalColorTable + 1) * 3;
                }
                while (base.GetByte(offset) != 0x3B)
                {
                    if (base.GetByte(offset) == 0x21)
                    {
                        switch (base.GetByte(offset + 1))
                        {
                            case 0xF9:
                                offset = checkGraphicCtrlEx(offset);
                                break;
                            case 0xFE:
                                offset = checkCommentEx(offset);
                                break;
                            case 0x01:
                                offset = checkPlainTextEx(offset);
                                break;
                            case 0xFF:
                                offset = checkApplEx(offset);
                                break;
                            default:
                                throw new NotifyException("ファイルは画像ファイルではありません。");
                        }
                    }
                    else if (base.GetByte(offset) == 0x2C)
                    {
                        offset = checkImageDescriptor(offset);
                    }
                    else
                    {
                        throw new NotifyException("ファイルは画像ファイルではありません。");
                    }

                }

                _blocks.Add(BlockType.Trailer, new List<uint> { offset });
                if (base.SourceLength != ++offset)
                {
                    _note.Add(string.Format("ファイルの最後に {0} バイトのゴミが付いています。",
                        base.SourceLength - offset));
                }
            }
            catch (NotifyException e)
            {
                base.Messages.Add(e.Message);
                return false;
            }
            catch (OutOfRangeException)
            {
                base.Messages.Add("ファイルは画像ファイルではありません。");
                return false;
            }

            makeCheckMessage();
            return true;
        }

        private UInt32 checkHeader(UInt32 offset)
        {
            var current = offset;
            if (base.GetByte(current++) != 0x47 || base.GetByte(current++) != 0x49 || base.GetByte(current++) != 0x46)
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }
            if (base.GetByte(current) == 0x38 && base.GetByte(current + 1) == 0x37 && base.GetByte(current + 2) == 0x61)
            {
                Version = VersionType.GIF87a;
            }
            else if (base.GetByte(current) == 0x38 && base.GetByte(current + 1) == 0x39 && base.GetByte(current + 2) == 0x61)
            {
                Version = VersionType.GIF89a;
            }
            else
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }

            _blocks.Add(BlockType.Header, new List<UInt32> { offset });
            return current + HEADER_VERSION_LENGTH;
        }

        private UInt32 checkLogicalScreenDescriptor(UInt32 offset)
        {
            _blocks.Add(BlockType.LogicalScreenDescriptor, new List<UInt32> { offset });
            var packedFields = base.GetByte(offset + LSD_PACKED_FIELDS_OFFSET);
            var globalColorTableFlag = (packedFields >> 7);
            _hasGlobalColorTable = false;
            if (globalColorTableFlag == 1)
            {
                _hasGlobalColorTable = true;
                _sizeOfGlobalColorTable = (byte)(packedFields & 0x07);
            }

            return offset + LOGICAL_SCREEN_DESCRIPTOR_LENGTH;
        }

        private UInt32 checkApplEx(UInt32 offset)
        {
            if (_blocks.ContainsKey(BlockType.ApplicationExtension))
            {
                _blocks[BlockType.ApplicationExtension].Add(offset);
            }
            else
            {
                _blocks.Add(BlockType.ApplicationExtension, new List<uint> { offset });
            }
            offset += EXTENSION_INTRO_LABEL_LENGTH;
            var blockLegth = base.GetByte(offset++);
            offset += blockLegth;
            while (base.GetByte(offset) != 0x00)
            {
                var subBlockLength = base.GetByte(offset++);
                offset += subBlockLength;
            }
            offset++;

            return offset;
        }

        private UInt32 checkGraphicCtrlEx(UInt32 offset)
        {
            if (_blocks.ContainsKey(BlockType.GraphicControlExtension))
            {
                _blocks[BlockType.GraphicControlExtension].Add(offset);
            }
            else
            {
                _blocks.Add(BlockType.GraphicControlExtension, new List<uint> { offset });
            }
            offset += EXTENSION_INTRO_LABEL_LENGTH;
            var blockLegth = base.GetByte(offset++);
            if (blockLegth != GRAPHIC_CTRL_EX_BLOCK_LENGTH)
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }
            offset += blockLegth;
            if (base.GetByte(offset++) != 0x00)
            {
                throw new NotifyException("ファイルは画像ファイルではありません。");
            }

            return offset;
        }

        private UInt32 checkCommentEx(UInt32 offset)
        {
            if (_blocks.ContainsKey(BlockType.CommentExtension))
            {
                _blocks[BlockType.CommentExtension].Add(offset);
            }
            else
            {
                _blocks.Add(BlockType.CommentExtension, new List<uint> { offset });
            }
            offset += EXTENSION_INTRO_LABEL_LENGTH;
            while (base.GetByte(offset) != 0x00)
            {
                var subBlockLength = base.GetByte(offset++);
                offset += subBlockLength;
            }
            offset++;

            return offset;
        }

        private UInt32 checkPlainTextEx(UInt32 offset)
        {
            if (_blocks.ContainsKey(BlockType.PlainTextExtension))
            {
                _blocks[BlockType.PlainTextExtension].Add(offset);
            }
            else
            {
                _blocks.Add(BlockType.PlainTextExtension, new List<uint> { offset });
            }
            offset += EXTENSION_INTRO_LABEL_LENGTH;
            var blockLegth = base.GetByte(offset++);
            offset += blockLegth;
            while (base.GetByte(offset) != 0x00)
            {
                var subBlockLength = base.GetByte(offset++);
                offset += subBlockLength;
            }
            offset++;

            return offset;
        }

        private UInt32 checkImageDescriptor(UInt32 offset)
        {
            while (base.GetByte(offset) == 0x2C)
            {
                if (!_blocks.ContainsKey(BlockType.ImageDescriptor))
                {
                    _blocks.Add(BlockType.ImageDescriptor, new List<uint> { offset });
                }
                else
                {
                    _blocks[BlockType.ImageDescriptor].Add(offset);
                }
                var packedFields = base.GetByte(offset + ID_PACKED_FIELDS_OFFSET);
                UInt32 sizeOfLocalColorTable = 0;
                var localColorTableFlag = (packedFields >> 7);
                offset += IMAGE_DESCRIPTOR_LENGTH;
                // ここからローカル カラー テーブル
                if (localColorTableFlag == 1)
                {
                    sizeOfLocalColorTable = (byte)(packedFields & 0x07);
                    offset += (UInt32)Math.Pow(2, sizeOfLocalColorTable + 1) * 3;
                }
                // ここからイメージデータ
                offset++; // LZW Minimum Code Size を読み飛ばし
                while (base.GetByte(offset) != 0x00)
                {
                    var subBlockLength = base.GetByte(offset++);
                    offset += subBlockLength;
                }
                offset++;
            }

            return offset;
        }

        private void makeCheckMessage()
        {
            base.Messages.Add("【GIF ファイル構成情報】");
            base.Messages.Add(string.Format("GIF Version: {0}", Version));
            foreach (var n in _blocks.Keys)
            {
                base.Messages.Add(string.Format("{0} 個の{1}ブロックを持っています。",
                    _blocks[n].Count, n));
            }

            foreach (var n in _note)
            {
                base.Messages.Add(n);
            }
            _note.Clear();
        }

        protected override void analize()
        {
            var ls = new GifLogicalScreenInfo(base.Source, _blocks[BlockType.LogicalScreenDescriptor].First());
            base.Messages.Add(string.Format("{0}【LogicalScreenDescriptor 情報】", Environment.NewLine));
            base.Messages.Add(string.Format("Logical Screen size: {0} x {1}", ls.LogicalScreenWidth, ls.LogicalScreenHeight));
            base.Messages.Add(string.Format("Global Color Table のエントリ数: {0}", ls.ColorResolution));

            if (_blocks.ContainsKey(BlockType.ApplicationExtension))
            {
                var ae = new GifAppliExtInfo(base.Source, _blocks[BlockType.ApplicationExtension].First());
                base.Messages.Add(string.Format("{0}【ApplicationExtension 情報】", Environment.NewLine));
                base.Messages.Add(string.Format("アニメーション GIF: {0}", ae.IsAnimatedGIF));
                base.Messages.Add(string.Format("アニメーション繰り返し回数: {0} (0は無限ループ)", ae.AnimationRepetitionTimes));
            }
        }

        public enum VersionType
        {
            GIF87a,
            GIF89a
        }

        public enum BlockType
        {
            Header,
            LogicalScreenDescriptor,
            GlobalColorTable,
            ImageDescriptor,
            LocalColorTable,
            TableBasedImageData,
            GraphicControlExtension,
            CommentExtension,
            PlainTextExtension,
            ApplicationExtension,
            Trailer,
        }
    }
}
