﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading.Tasks;

namespace GetJpegInfo.Graphics
{
    public abstract class GraphicsInfoBase
    {
        public static readonly UInt32 INT16_OFFSET = 2;
        public static readonly UInt32 INT32_OFFSET = 4;
        public static readonly UInt32 INT64_OFFSET = 8;

        public GraphicsInfoBase() { }
        public GraphicsInfoBase(byte[] source, IList<string> messages, PicFormat format)
        {
            Init(source, messages, format);
        }

        private byte[] _source = null;
        /// <summary>
        /// 画像データを格納する byte 配列
        /// </summary>
        protected byte[] Source
        {
            get { return _source; }
        }

        private bool _isInit = false;
        /// <summary>
        /// データが初期化されているかを取得します。
        /// </summary>
        protected bool IsInit
        {
            get { return _isInit; }
        }

        /// <summary>
        /// 画像データの長さを取得します。
        /// </summary>
        protected UInt32 SourceLength
        {
            get { return (uint)_source.Length; }
        }

        private IList<string> _messages = null;
        /// <summary>
        /// 画像ファイルから取得したデータのリストを取得します。
        /// </summary>
        protected IList<string> Messages
        {
            get { return _messages; }
        }

        private PicFormat _imageFormat = PicFormat.Unknown;
        /// <summary>
        /// 画像ファイルの種別を取得します。
        /// </summary>
        public PicFormat ImageFormat
        {
            get { return _imageFormat; }
        }

        /// <summary>
        /// 画像ファイルから取得したデータのリストを文字列にして取得します。
        /// </summary>
        /// <returns></returns>
        public string GetMessage()
        {
            return string.Join(Environment.NewLine, Messages);
        }

        /// <summary>
        /// 画像ファイルが JFIF のサムネイルを持っているかを取得します。
        /// </summary>
        public bool HasJfifThumbnail { get; protected set; }

        /// <summary>
        /// JFIF のサムネイルの幅を取得します。
        /// </summary>
        public int JfifThumbnailWidth { get; protected set; }

        /// <summary>
        /// JFIF のサムネイルの高さを取得します。
        /// </summary>
        public int JfifThumbnailHeight { get; protected set; }

        /// <summary>
        /// JFIF 形式のサムネイルのバイト配列を取得します。
        /// </summary>
        public byte[] JfifThumbnail { get; protected set; }

        /// <summary>
        /// 画像ファイルが Exif のサムネイルを持っているかを取得します。
        /// </summary>
        public bool HasExifThumbnail { get; protected set; }

        /// <summary>
        /// Exif 形式のサムネイルのバイト配列を取得します。
        /// </summary>
        public byte[] ExifThumbnail { get; protected set; }

        /// <summary>
        /// GraphicsInfoBase クラスのインスタンスを取得します。
        /// </summary>
        /// <param name="filePath">画像ファイルへのパス</param>
        /// <returns></returns>
        public static async Task<GraphicsInfoBase> GetGraphicsInfoBase(string filePath)
        {
            GraphicsInfoBase result = null;
            var messages = new List<string>();
            var imageFormat = PicFormat.Unknown;
            byte[] buff = null;
            await Task.Run(async () =>
            {
                using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
                using (var reader = new BinaryReader(fs))
                {
                    buff = new byte[16];
                    var readLength = await Task.Factory.StartNew<int>(() => { return reader.Read(buff, 0, buff.Length); });
                    if (readLength < buff.Length)
                    {
                        throw new NotifyException("ファイルは画像ファイルではありません。");
                    }

                    if (buff[0] == 0x89 && buff[1] == 0x50 && buff[2] == 0x4E && buff[3] == 0x47 && buff[4] == 0x0D && buff[5] == 0xA)
                    {
                        messages.Add("画像ファイルは PNG です。");
                        imageFormat = PicFormat.PNG;
                        fs.Position = 0;
                        buff = new byte[fs.Length];
                        await Task.Factory.StartNew<int>(() => { return reader.Read(buff, 0, buff.Length); });
                    }
                    else if (buff[0] == 0xFF && buff[1] == 0xD8)
                    {
                        messages.Add("画像ファイルは JPEG です。");
                        imageFormat = PicFormat.JPEG;
                        fs.Position = 0;
                        buff = new byte[fs.Length];
                        await Task.Factory.StartNew<int>(() => { return reader.Read(buff, 0, buff.Length); });
                    }
                    else if (buff[0] == 0x47 && buff[1] == 0x49 && buff[2] == 0x46)
                    {
                        messages.Add("画像ファイルは GIF です。");
                        imageFormat = PicFormat.GIF;
                        fs.Position = 0;
                        buff = new byte[fs.Length];
                        await Task.Factory.StartNew<int>(() => { return reader.Read(buff, 0, buff.Length); });
                    }
                    else if ((buff[0] == 0x4D && buff[1] == 0x4D && buff[2] == 0x00 && buff[3] == 0x2A) ||
                        (buff[0] == 0x49 && buff[1] == 0x49 && buff[2] == 0x2A && buff[3] == 0x00))
                    {
                        messages.Add("画像ファイルは TIFF です。");
                        imageFormat = PicFormat.TIFF;
                        fs.Position = 0;
                        buff = new byte[fs.Length];
                        await Task.Factory.StartNew<int>(() => { return reader.Read(buff, 0, buff.Length); });
                    }
                    else
                    {
                        throw new NotifyException("ファイルは、JPEG, PNG, GIF, TIFF 形式の画像ファイルではありません。");
                    }
                }
            });

            switch (imageFormat)
            {
                case PicFormat.JPEG:
                    result = new JpegInfo(buff, messages, imageFormat);
                    break;
                case PicFormat.PNG:
                    result = new PngInfo(buff, messages, imageFormat);
                    break;
                case PicFormat.TIFF:
                    result = new TiffInfo(buff, messages, imageFormat);
                    break;
                case PicFormat.GIF:
                    result = new GifInfo(buff, messages, imageFormat);
                    break;
            }

            return result;
        }

        /// <summary>
        /// 画像ファイルの形式を確認し情報を取得します。
        /// </summary>
        public void ImageCheck()
        {

            if (check())
            {
                analize();
            }
        }

        /// <summary>
        /// データを初期化します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="message"></param>
        /// <param name="format"></param>
        protected void Init(byte[] source, IList<string> message, PicFormat format)
        {
            if (_isInit)
            {
                throw new ApplicationException("既に初期化済み。");
            }

            _source = source;
            _messages = message;
            _isInit = true;
            _imageFormat = format;
        }

        /// <summary>
        /// 画像ファイルの形式を確認します。
        /// </summary>
        /// <returns></returns>
        protected abstract bool check();
        /// <summary>
        /// 画像ファイルから情報を取得します。
        /// </summary>
        protected abstract void analize();

        /// <summary>
        /// 画像データの byte 配列から byte データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <returns></returns>
        protected byte GetByte(UInt32 sourceIndex)
        {
            return GetByte(Source, sourceIndex);
        }
        /// <summary>
        /// byte 配列から byte データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <returns></returns>
        public static byte GetByte(byte[] source, UInt32 sourceIndex)
        {
            if (sourceIndex >= source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            return source[sourceIndex];
        }

        /// <summary>
        /// 画像データの byte 配列から sbyte データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <returns></returns>
        protected sbyte GetSByte(UInt32 sourceIndex)
        {
            return GetSByte(Source, sourceIndex);
        }
        /// <summary>
        /// byte 配列から sbyte データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <returns></returns>
        public static sbyte GetSByte(byte[] source, UInt32 sourceIndex)
        {
            if (sourceIndex >= source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            return unchecked((sbyte)source[sourceIndex]);
        }

        /// <summary>
        /// 画像データの byte 配列から UInt16 データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        protected UInt16 GetUInt16(UInt32 sourceIndex, Endianness sourceEndian)
        {
            return GetUInt16(Source, sourceIndex, sourceEndian);
        }
        /// <summary>
        /// byte 配列から UInt16 データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        public static UInt16 GetUInt16(byte[] source, UInt32 sourceIndex, Endianness sourceEndian)
        {
            if (sourceIndex + INT16_OFFSET > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            var temp = new byte[INT16_OFFSET];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            if (BitConverter.IsLittleEndian != (sourceEndian == Endianness.Little))
            {
                Array.Reverse(temp);
            }
            return BitConverter.ToUInt16(temp, 0);
        }

        /// <summary>
        /// 画像データの byte 配列から Int16 データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        protected Int16 GetInt16(UInt32 sourceIndex, Endianness sourceEndian)
        {
            return GetInt16(Source, sourceIndex, sourceEndian);
        }
        /// <summary>
        /// byte 配列から Int16 データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        public static Int16 GetInt16(byte[] source, UInt32 sourceIndex, Endianness sourceEndian)
        {
            if (sourceIndex + INT16_OFFSET > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            var temp = new byte[INT16_OFFSET];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            if (BitConverter.IsLittleEndian != (sourceEndian == Endianness.Little))
            {
                Array.Reverse(temp);
            }

            return BitConverter.ToInt16(temp, 0);
        }

        /// <summary>
        /// 画像データの byte 配列から UInt32 データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        protected UInt32 GetUInt32(UInt32 sourceIndex, Endianness sourceEndian)
        {
            return GetUInt32(Source, sourceIndex, sourceEndian);
        }
        /// <summary>
        /// byte 配列から UInt32 データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        public static UInt32 GetUInt32(byte[] source, UInt32 sourceIndex, Endianness sourceEndian)
        {
            if (sourceIndex + INT32_OFFSET > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            var temp = new byte[INT32_OFFSET];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            if (BitConverter.IsLittleEndian != (sourceEndian == Endianness.Little))
            {
                Array.Reverse(temp);
            }

            return BitConverter.ToUInt32(temp, 0);
        }

        /// <summary>
        /// 画像データの byte 配列から UInt32 データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        protected Int32 GetInt32(UInt32 sourceIndex, Endianness sourceEndian)
        {
            return GetInt32(Source, sourceIndex, sourceEndian);
        }
        /// <summary>
        /// byte 配列から UInt32 データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        public static Int32 GetInt32(byte[] source, UInt32 sourceIndex, Endianness sourceEndian)
        {
            if (sourceIndex + INT32_OFFSET > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            var temp = new byte[INT32_OFFSET];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            if (BitConverter.IsLittleEndian != (sourceEndian == Endianness.Little))
            {
                Array.Reverse(temp);
            }

            return BitConverter.ToInt32(temp, 0);
        }

        /// <summary>
        /// 画像データの byte 配列から float データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        protected float GetSingle(UInt32 sourceIndex, Endianness sourceEndian)
        {
            return GetSingle(Source, sourceIndex, sourceEndian);
        }
        /// <summary>
        /// byte 配列から float データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        public static float GetSingle(byte[] source, UInt32 sourceIndex, Endianness sourceEndian)
        {
            if (sourceIndex + INT32_OFFSET > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            var temp = new byte[INT32_OFFSET];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            if (BitConverter.IsLittleEndian != (sourceEndian == Endianness.Little))
            {
                Array.Reverse(temp);
            }

            return BitConverter.ToSingle(temp, 0);
        }

        /// <summary>
        /// 画像データの byte 配列から double データを取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        protected double GetDouble(UInt32 sourceIndex, Endianness sourceEndian)
        {
            return GetDouble(Source, sourceIndex, sourceEndian);
        }
        /// <summary>
        /// byte 配列から double データを取得します。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="sourceEndian"></param>
        /// <returns></returns>
        public static double GetDouble(byte[] source, UInt32 sourceIndex, Endianness sourceEndian)
        {
            if (sourceIndex + INT64_OFFSET > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }
            var temp = new byte[INT64_OFFSET];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            if (BitConverter.IsLittleEndian != (sourceEndian == Endianness.Little))
            {
                Array.Reverse(temp);
            }

            return BitConverter.ToDouble(temp, 0);
        }

        /// <summary>
        /// 画像データの byte 配列から文字列を取得します。
        /// 取得する文字列は null が現れるか 文字列長が length になるまでです。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="length"></param>
        /// <returns></returns>
        protected string GetStringUntilNull(UInt32 sourceIndex, UInt32 length)
        {
            return GetStringUntilNull(Source, sourceIndex, length);
        }
        /// <summary>
        /// byte 配列から文字列を取得します。
        /// 取得する文字列は null が現れるか 文字列長が length になるまでです。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="length"></param>
        /// <returns></returns>
        public static string GetStringUntilNull(byte[] source, UInt32 sourceIndex, UInt32 length)
        {
            string utf8String;
            GetStringUntilNull(source, sourceIndex, length, out utf8String);
            return utf8String;
        }
        /// <summary>
        /// 画像データの byte 配列から文字列と取得した文字列長を取得します。
        /// 取得する文字列は null が現れるか 文字列長が length になるまでです。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="length"></param>
        /// <param name="utf8String">取得した文字列を格納します。</param>
        /// <returns>取得した文字列長</returns>
        protected UInt32 GetStringUntilNull(UInt32 sourceIndex, UInt32 length, out string utf8String)
        {
            return GetStringUntilNull(Source, sourceIndex, length, out utf8String);
        }
        /// <summary>
        /// byte 配列から文字列と取得した文字列長を取得します。
        /// 取得する文字列は null が現れるか 文字列長が length になるまでです。
        /// </summary>
        /// <param name="source"></param>
        /// <param name="sourceIndex"></param>
        /// <param name="length"></param>
        /// <param name="utf8String">取得した文字列を格納します。</param>
        /// <returns>取得した文字列長</returns>
        public static UInt32 GetStringUntilNull(byte[] source, UInt32 sourceIndex, UInt32 length, out string utf8String)
        {
            if (sourceIndex + length > source.Length)
            {
                throw new OutOfRangeException("指定されたオフセット値が範囲外となっています。");
            }

            for (UInt32 i = 0; i < length; ++i)
            {
                if (source[sourceIndex + i] == 0x00)
                {
                    length = i;
                    break;
                }
            }
            var temp = new byte[length];
            Array.Copy(source, sourceIndex, temp, 0, temp.Length);
            utf8String = Encoding.UTF8.GetString(temp);

            return length;
        }

        /// <summary>
        /// 画像データの byte 配列から指定された範囲の要素を取得します。
        /// </summary>
        /// <param name="sourceIndex"></param>
        /// <param name="length"></param>
        /// <returns></returns>
        protected byte[] GetBytes(UInt32 sourceIndex, UInt32 length)
        {
            var result = new byte[length];
            Array.Copy(Source, sourceIndex, result, 0, length);
            return result;
        }
    }

    /// <summary>
    /// エンディアンの種別を表す列挙型
    /// </summary>
    public enum Endianness
    {
        Big,
        Little
    }

    /// <summary>
    /// 画像ファイルの形式を表す列挙型
    /// </summary>
    public enum PicFormat
    {
        PNG,
        JPEG,
        GIF,
        TIFF,
        BMP,
        Unknown
    }
}
