﻿using System.Linq;

namespace GetJpegInfo.Graphics
{
    /// <summary>
    /// JPEG Exif 形式の GPS IFD の情報を取得・保持します。
    /// </summary>
    public class JpegGpsInfo : ExifTags
    {
        public JpegGpsInfo() { }

        private string _gpsLatitudeRef = null;
        private Rational[] _gpsLatitude = null;
        public string GpsLatitude
        {
            get
            {
                if (_gpsLatitude == null)
                {
                    return null;
                }
                string direction;
                if (_gpsLatitudeRef == "N")
                {
                    direction = "北緯";
                }
                else if (_gpsLatitudeRef == "S")
                {
                    direction = "南緯";
                }
                else
                {
                    throw new NotifyException("GPS データの方位が不正です。");
                }
                var val = string.Format("{0} {1}", direction, gpsLocation(_gpsLatitude));
                return _gpsLatitude != null ?
                    base.displayTag(TiffTag.GPSLatitude, val) : null;
            }
        }

        private string _gpsLongitudeRef = null;
        private Rational[] _gpsLongitude = null;
        public string GpsLongitude
        {
            get
            {
                if (_gpsLongitude == null)
                {
                    return null;
                }
                string direction;
                if (_gpsLongitudeRef == "E")
                {
                    direction = "東経";
                }
                else if (_gpsLongitudeRef == "W")
                {
                    direction = "西経";
                }
                else
                {
                    throw new NotifyException("GPS データの方位が不正です。");
                }
                var val = string.Format("{0} {1}", direction, gpsLocation(_gpsLongitude));
                return _gpsLongitude != null ?
                    base.displayTag(TiffTag.GPSLongitude, val) : null;
            }
        }

        private byte _gpsAltitudeRef = 0;
        private Rational[] _gpsAltitude = null;
        public string GpsAltitude
        {
            get
            {
                if (_gpsAltitude == null)
                {
                    return null;
                }
                int sign = 0;
                if (_gpsAltitudeRef == 0)
                {
                    sign = 1;
                }
                else if (_gpsAltitudeRef == 1)
                {
                    sign = -1;
                }
                else
                {
                    throw new NotifyException("GPS データの高度符号値が不正です。");
                }
                var val = computeRational(_gpsAltitude[0]) * sign;

                return _gpsAltitude != null ?
                    base.displayTag(TiffTag.GPSAltitude, string.Format("海抜 {0:#,0.##}m", val)) : null;
            }
        }

        private Rational[] _gpsTimeStamp = null;
        public string GpsTimeStamp
        {
            get
            {
                if (_gpsTimeStamp == null)
                {
                    return null;
                }
                if (_gpsTimeStamp.Count() != 3)
                {
                    throw new NotifyException("GPS タイムスタンプのデータ数が不正です。");
                }
                return _gpsTimeStamp != null ?
                    base.displayTag(TiffTag.GPSTimeStamp, string.Format("{0}:{1}:{2} UTC",
                    computeRational(_gpsTimeStamp[0]),
                    computeRational(_gpsTimeStamp[1]),
                    computeRational(_gpsTimeStamp[2]))) : null;
            }
        }

        public override void AddTagInfo(TiffTag tagNo, TiffTagData tagData)
        {
            switch (tagNo)
            {
                case TiffTag.GPSLatitudeRef:
                    _gpsLatitudeRef = tagData.AsciiData;
                    break;
                case TiffTag.GPSLatitude:
                    _gpsLatitude = tagData.RationalData;
                    break;
                case TiffTag.GPSLongitudeRef:
                    _gpsLongitudeRef = tagData.AsciiData;
                    break;
                case TiffTag.GPSLongitude:
                    _gpsLongitude = tagData.RationalData;
                    break;
                case TiffTag.GPSAltitudeRef:
                    _gpsAltitudeRef = tagData.ByteData.First();
                    break;
                case TiffTag.GPSAltitude:
                    _gpsAltitude = tagData.RationalData;
                    break;
                case TiffTag.GPSTimeStamp:
                    _gpsTimeStamp = tagData.RationalData;
                    break;
                default:
                    base.AddMessage(string.Format("{0}: {1}", (TiffTag)tagNo, tagData));
                    break;
            }
        }

        private string gpsLocation(Rational[] rational)
        {
            var amount = rational.Count();
            float[] vals = new float[amount];
            for (var i = 0; i < amount; ++i)
            {
                vals[i] = computeRational(rational[i]);
            }

            return string.Format("{0}度 {1}分 {2}秒", vals[0], vals[1], vals[2]);
        }

        private float computeRational(Rational rational)
        {
            return rational.Numerator / (float)rational.Denominator;
        }
    }
}
