﻿using System;
using System.Collections.Generic;

namespace GetJpegInfo.Graphics
{
    public class TiffInfo : GraphicsInfoBase
    {
        private const UInt32 HEADER_LENGTH = 8;
        private const UInt32 IFD_ENTRY_LEGTH = 12;
        private Endianness _dataEndian = Endianness.Big;
        private List<UInt32> _ifds = new List<uint>();
        private List<string> _note = new List<string>();

        public TiffInfo() { }
        public TiffInfo(byte[] source, IList<string> message, PicFormat format) : base(source, message, format) { }

        protected override bool check()
        {
            UInt32 offset = 0;
            try
            {
                offset = checkHeader(offset);
                while (offset != 0)
                {
                    offset = checkIfd(offset);
                }
            }
            catch (NotifyException e)
            {
                base.Messages.Add(e.Message);
                return false;
            }
            catch (OutOfRangeException)
            {
                base.Messages.Add("ファイルは画像ファイルではありません。");
                return false;
            }

            makeCheckMessage();
            return true;
        }

        protected override void analize()
        {
            UInt32 offset = 0;
            UInt16 numberEntries = 0;
            foreach (var n in _ifds)
            {
                base.Messages.Add("【IFD タグ情報】");
                var tiffTags = new TiffTagInfo();
                offset = n;
                numberEntries = base.GetUInt16(offset, _dataEndian);
                offset += GraphicsInfoBase.INT16_OFFSET;
                for (var i = 0; i < numberEntries; ++i)
                {
                    analizeIfdEntry(offset, tiffTags);
                    offset += IFD_ENTRY_LEGTH;
                }
                base.Messages.Add(string.Format("Picture size: {0} x {1}", tiffTags.ImageWidth, tiffTags.ImageHeight));
                base.Messages.Add(string.Format("Color depth: {0} bits per pixel", tiffTags.ColorDepth));
                base.Messages.Add(string.Format("PhotometricInterpretation: {0}", tiffTags.PhotometricInterpretation));
                base.Messages.Add(string.Format("Compression: {0}", tiffTags.Compression));
                foreach (var tm in tiffTags.TagInfos)
                {
                    base.Messages.Add(tm);
                }
            }
        }

        private UInt32 checkHeader(UInt32 offset)
        {
            if (offset != 0 || offset + HEADER_LENGTH > base.SourceLength)
            {
                throw new NotifyException("画像ファイルではありません。");
            }

            if (base.GetByte(offset) == 0x49 && base.GetByte(offset + 1) == 0x49)
            {
                _dataEndian = Endianness.Little;
            }
            else if (base.GetByte(offset) != 0x4D || base.GetByte(offset + 1) != 0x4D)
            {
                throw new NotifyException("画像ファイルではありません。");
            }
            offset += GraphicsInfoBase.INT16_OFFSET;
            if (base.GetUInt16(offset, _dataEndian) != 42u) // Tiff 識別子
            {
                throw new NotifyException("画像ファイルではありません。");
            }
            offset += GraphicsInfoBase.INT16_OFFSET;
            offset = base.GetUInt32(offset, _dataEndian);
            if (offset >= base.SourceLength)
            {
                throw new NotifyException("1st IFD へのオフセット値が不正です。");
            }

            return offset;
        }

        private UInt32 checkIfd(UInt32 offset)
        {
            _ifds.Add(offset); // IFD へのオフセット値を記録
            var numberEntries = base.GetUInt16(offset, _dataEndian);
            offset += GraphicsInfoBase.INT16_OFFSET;
            if (numberEntries * IFD_ENTRY_LEGTH + offset + GraphicsInfoBase.INT32_OFFSET > base.SourceLength)
            {
                throw new NotifyException("IFD のエントリ数の値が不正です。");
            }
            offset += numberEntries * IFD_ENTRY_LEGTH;
            offset = base.GetUInt32(offset, _dataEndian);
            if (offset >= base.SourceLength)
            {
                throw new NotifyException("次の IFD へのオフセット値が不正です。");
            }

            return offset;
        }

        private void makeCheckMessage()
        {
            base.Messages.Add("【TIFF ファイル構成情報】");

            base.Messages.Add(string.Format("{0} 個の IFD を持っています。", _ifds.Count));

            foreach (var n in _note)
            {
                base.Messages.Add(n);
            }
            _note.Clear();
        }

        private void analizeIfdEntry(UInt32 offset, TiffTagInfo taginfo)
        {
            UInt32 exifIFDPointer = 0;
            UInt32 gpsIFDPointer = 0;
            var tagNo = base.GetUInt16(offset, _dataEndian);
            offset += GraphicsInfoBase.INT16_OFFSET;
            var tagType = base.GetUInt16(offset, _dataEndian);
            offset += GraphicsInfoBase.INT16_OFFSET;
            var tagValAmount = base.GetUInt32(offset, _dataEndian);
            offset += GraphicsInfoBase.INT32_OFFSET;
            var tagValueIndex = offset;

            if((TiffTag)tagNo == TiffTag.GPSInfoIFDPointer)
            {
                gpsIFDPointer = base.GetUInt32(tagValueIndex, _dataEndian);
                taginfo.AddMessage("《GPS IFD を持っています。》");
            }
            else if ((TiffTag)tagNo == TiffTag.ExifIFDPointer)
            {
                exifIFDPointer = base.GetUInt32(tagValueIndex, _dataEndian);
                taginfo.AddMessage("《EXIF IFD を持っています。》");
            }
            else
            {
                var tagData = new TiffTagData(base.Source, tagValAmount, tagValueIndex, _dataEndian, (TagType)tagType);
                taginfo.AddTagInfo((TiffTag)tagNo, tagData);
            }
        }
    }
}
