﻿using System;
using System.Text;
using System.ComponentModel;
using System.Windows;
using System.Windows.Input;
using System.Diagnostics;
using System.Globalization;

namespace Ipv6UniqueLocal
{
    public class MainWindowViewModel : INotifyPropertyChanged, IDataErrorInfo
    {
        #region Fields

        private string timeStamp;
        private string macAddr;
        private string eui64;
        private string seed;
        private string digest;
        private string globalId;
        private byte[] prefix = new byte[6];
        private string subnet;
        private string interfaceId;
        private byte[] uniqueLocal = new byte[16];
        private bool isPrefixCalculated;
        private CultureInfo selectedItem;
        private ICommand prefixGetCommand;
        private ICommand copyAddrCommand;
        private ICommand closeCommand;

        #endregion // Fields

        #region Constructor

        public MainWindowViewModel()
        {
            IsPrefixCalculated = false;
            Subnet = "0000";
            InterfaceId = "0000000000000000";
            SelectedItem = CultureInfo.CurrentUICulture;
        }

        #endregion // Constructor

        #region Properties

        public CultureInfo SelectedItem
        {
            get { return selectedItem; }
            set
            {
                selectedItem = (CultureInfo)value;
                OnPropertyChanged("SelectedItem");
                CultureResources.ChangeCulture(selectedItem);
            }
        }

        public string TimeStamp
        {
            get { return timeStamp; }
            private set
            {
                timeStamp = value;
                OnPropertyChanged("TimeStamp");
            }
        }
        public string MacAddr {
            get { return macAddr; }
            private set
            {
                macAddr = value;
                OnPropertyChanged("MacAddr");
            }
        }
        public string Eui64 {
            get { return eui64; }
            private set
            {
                eui64 = value;
                OnPropertyChanged("Eui64");
            }
        }
        public string Seed
        {
            get { return seed; }
            private set
            {
                seed = value;
                OnPropertyChanged("Seed");
            }
        }
        public string Digest
        {
            get { return digest; }
            private set
            {
                digest = value;
                OnPropertyChanged("Digest");
            }
        }
        public string GlobalId
        {
            get { return globalId; }
            private set
            {
                globalId = value;
                OnPropertyChanged("GlobalId");
            }
        }
        public string Prefix
        {
            get { return bytes2Hex(prefix); }
            private set
            {
                prefix = string2Bytes(value);
                OnPropertyChanged("Prefix");
                setUniqueLocal();
            }
        }
        public string Subnet
        {
            get { return subnet; }
            set
            {
                subnet = value;
                OnPropertyChanged("Subnet");
                setUniqueLocal();
            }
        }
        public string InterfaceId
        {
            get { return interfaceId; }
            set
            {
                interfaceId = value;
                OnPropertyChanged("InterfaceId");
                setUniqueLocal();
            }
        }
        public string UniqueLocal
        {
            get { return Ipv6Util.bytes2Ip6DisplayShortenString(uniqueLocal); }
            set
            {
                uniqueLocal = string2Bytes(value);
                OnPropertyChanged("UniqueLocal");
            }
        }

        public bool IsPrefixCalculated
        {
            get { return isPrefixCalculated; }
            set
            {
                isPrefixCalculated = value;
            }
        }

        public ICommand PrefixGetCommand
        {
            get
            {
                if (prefixGetCommand == null)
                {
                    prefixGetCommand = new RelayCommand(
                        param => prefixGetCommandExecute(),
                        param => prefixGetCommandCanExecute
                        );
                }
                return prefixGetCommand;
            }
        }

        public ICommand CopyAddrCommand
        {
            get
            {
                if (copyAddrCommand == null)
                {
                    copyAddrCommand = new RelayCommand(
                        param => copyAddrCommandExecute(),
                        param => copyAddrCommandCanExecute
                        );
                }
                return copyAddrCommand;
            }
        }

        public ICommand CloseCommand
        {
            get
            {
                if (closeCommand == null)
                {
                    closeCommand = new RelayCommand(
                        param => closeCommandExecute()
                        );
                }
                return closeCommand;
            }
        }

        #endregion //Properties

        #region Perform availability evaluation and implementation of the command

        // PrefixGet ボタンクリックの実行メソッド
        private void prefixGetCommandExecute()
        {
            var timeStamp = Ipv6Util.Datetime2NtpTimeStamp(DateTime.UtcNow);
            var eui64 = Ipv6Util.GetEui64();
            var seed = Ipv6Util.GetSeed(timeStamp, eui64);
            var digest = Ipv6Util.GetSha1Digest(seed);
            var glovalId = Ipv6Util.GetGlobalId(digest);
            var prefix = Ipv6Util.GetRoutingPrefixUniqueLocalUnicast(glovalId);
            TimeStamp = ulong2Hex(timeStamp);
            MacAddr = bytes2Hex(Ipv6Util.GetMacAddress());
            Eui64 = bytes2Hex(eui64);
            Seed = bytes2Hex(seed);
            Digest = bytes2Hex(digest);
            GlobalId = bytes2Hex(glovalId);
            Prefix = bytes2Hex(prefix);

            IsPrefixCalculated = true;
        }
        // PrefixGet ボタンのクリック可否
        private bool prefixGetCommandCanExecute
        {
            get { return !IsPrefixCalculated && IsValid; }
        }

        // CopyAddr ボタンクリックの実行メソッド
        private void copyAddrCommandExecute()
        {
            Clipboard.SetText(UniqueLocal);
        }
        // PrefixGet ボタンのクリック可否
        private bool copyAddrCommandCanExecute
        {
            get { return IsPrefixCalculated && IsValid; }
        }

        // Close ボタンクリックの実行メソッド
        private void closeCommandExecute()
        {
            App.Current.Shutdown();
        }

        #endregion // Perform availability evaluation and implementation of the command

        #region Private Helpers

        private static string bytes2Hex(byte[] bytes)
        {
            var val = new StringBuilder();
            foreach (var n in bytes)
            {
                val.Append(string.Format("{0:X2}", n));
            }
            return val.ToString();
        }

        private static string ulong2Hex(ulong source)
        {
            var bytes = BitConverter.GetBytes(source);
            if (BitConverter.IsLittleEndian)    // ホストバイトオーダーがリトルエンディアンだったら
                Array.Reverse(bytes);           // ビッグエンディアンへ変換する
            return bytes2Hex(bytes);
        }

        private static byte[] string2Bytes(string source)
        {
            var length = source.Length / 2;
            var bytes = new byte[length];
            for (var i = 0; i < length; i++)
            {
                bytes[i] = Convert.ToByte(source.Substring(i * 2, 2), 16);
            }
            return bytes;
        }

        private bool isHex(string test)
        {
            var patternStr = @"^[0-9a-fA-F]+$";
            var rgx = new System.Text.RegularExpressions.Regex(patternStr);
            var m = rgx.Match(test);
            return m.Success;
        }

        private void setUniqueLocal()
        {
            UniqueLocal = Prefix + Subnet + InterfaceId;
        }

        #endregion // Private Helpers

        #region INotifyPropertyChanged Members

        public event PropertyChangedEventHandler PropertyChanged;

        protected void OnPropertyChanged(string propertyName)
        {
            var handler = PropertyChanged;
            if (handler != null)
                handler(this, new PropertyChangedEventArgs(propertyName));
        }

        #endregion // INotifyPropertyChanged Members

        #region IDataErrorInfo Members

        public string Error
        {
            get { return null; }
        }

        public string this[string propertyName]
        {
            get {
                return GetValidationError(propertyName);
            }
        }

        #endregion // IDataErrorInfo Members

        #region Validation

        public bool IsValid
        {
            get
            {
                foreach (var property in ValidatedProperties)
                {
                    if (!string.IsNullOrEmpty(GetValidationError(property)))
                        return false;
                }

                return true;
            }
        }

        private static readonly string[] ValidatedProperties =
        {
            "Subnet",
            "InterfaceId",
        };

        private string GetValidationError(string propertyName)
        {
            if (Array.IndexOf(ValidatedProperties, propertyName) < 0) return null;

            string errorMessage = null;

            switch (propertyName)
            {
                case "Subnet":
                    errorMessage = ValidateSubnet();
                    break;
                case "InterfaceId":
                    errorMessage = ValidateInterfaceId();
                    break;
                default:
                    Debug.Fail("Validate property setting is incorrect: " + propertyName);
                    break;
            }

            return errorMessage;
        }

        private string ValidateSubnet()
        {
            var val = "";

            if (subnet.Length != 4)
                val = Properties.Resources.MainWindow_ErrMsg_SubnetLength;
            if (!isHex(subnet))
                val = Properties.Resources.MainWindow_ErrMsg_SubnetCharacters;
            return val;
        }

        private string ValidateInterfaceId()
        {
            var val = "";

            if (interfaceId.Length != 16)
                val = Properties.Resources.MainWindow_ErrMsg_InterfaceIDLength;
            if (!isHex(interfaceId))
                val = Properties.Resources.MainWindow_ErrMsg_InterfaceIDCharacters;
            return val;
        }

        #endregion // Validation
    }
}
