﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Text;
using System.Net.Sockets;
using System.Net.Security;
using System.Diagnostics;

namespace MakCraft.SmtpOverSsl
{
    /// <summary>
    /// 送信用のメールメッセージ
    /// </summary>
    public class SmtpMailMessage
    {
        /// <summary>
        /// 送信者のメールアドレス
        /// </summary>
        /// <exception cref="FormatException"></exception>
        public System.Net.Mail.MailAddress From { get; set; }

        /// <summary>
        /// 送信者のメールアドレス
        /// </summary>
        /// <exception cref="FormatException"></exception>
        public string StringFrom
        {
            get { return From.Address; }
            set { From = new System.Net.Mail.MailAddress(value); }
        }

        /// <summary>
        /// 送信先のメールアドレス
        /// </summary>
        /// <exception cref="FormatException"></exception>
        public System.Net.Mail.MailAddress To { get; set; }

        /// <summary>
        /// 送信先のメールアドレス
        /// </summary>
        /// <exception cref="FormatException"></exception>
        public string StringTo
        {
            get { return To.Address; }
            set { To = new System.Net.Mail.MailAddress(value); }
        }

        /// <summary>
        /// メールの件名
        /// </summary>
        public string Subject { get; set; }

        /// <summary>
        /// メールの本文
        /// </summary>
        public string Body { get; set; }
    }

    /// <summary>
    /// Smtp クラスのラッパー
    /// </summary>
    public static class SmtpMail
    {
        /// <summary>
        /// SMTP サーバーの名前
        /// </summary>
        public static string ServerName { get; set; }

        /// <summary>
        /// SMTP サーバーの受付ポート番号
        /// </summary>
        public static int ServerPort { get; set; }

        /// <summary>
        /// 本文のエンコーディングに使用する IANA に登録されている名前
        /// </summary>
        public static string MailEncoding { get; set; }

        /// <summary>
        /// 認証で用いるユーザー名
        /// </summary>
        public static string AuthUserName { get; set; }

        /// <summary>
        /// 認証で用いるパスワード
        /// </summary>
        public static string AuthPassword { get; set; }

        /// <summary>
        /// 認証方法
        /// </summary>
        public static Smtp.SmtpAuthMethod AuthMethod { get; set; }

        /// <summary>
        /// SMTP サーバーとの接続に SSL を使用するかどうかを指定
        /// </summary>
        public static bool EnableSsl { get; set; }

        /// <summary>
        /// メールを送信する
        /// </summary>
        /// <param name="mail">送信するメール</param>
        /// <exception cref="InvalidOperationException">
        /// Thrown when... プロパティに必須項目がセットされていないとき
        /// </exception>
        /// <exception cref="SocketException">
        /// Thrown when... サーバーとのコネクションが確立できなかったとき
        /// </exception>
        /// <exception cref="System.Security.Authentication.AuthenticationException">
        /// Thrown when... SSL 接続が確立できなかったとき
        /// </exception>
        /// <returns></returns>
        public static void Send(SmtpMailMessage mail)
        {
            var smtp = new Smtp
            {
                ServerName = ServerName,
                ServerPort = ServerPort,
                EnableSsl = EnableSsl,
                AuthUserName = AuthUserName,
                AuthPassword = AuthPassword,
                AuthMethod = AuthMethod,
                MailEncoding = MailEncoding
            };
            smtp.Send(mail);
            return;
        }
    }

    /// <summary>
    /// SMTP サーバーとの交信を行う
    /// </summary>
    public class Smtp
    {
        /// <summary>
        /// SMTP サーバーの名前
        /// </summary>
        public string ServerName { get; set; }

        /// <summary>
        /// SMTP サーバーの受付ポート番号
        /// </summary>
        public int ServerPort { get; set; }

        /// <summary>
        /// 本文のエンコーディングに使用する IANA に登録されている名前
        /// </summary>
        public string MailEncoding { get; set; }

        /// <summary>
        /// 認証で用いるユーザー名
        /// </summary>
        public string AuthUserName { get; set; }

        /// <summary>
        /// 認証で用いるパスワード
        /// </summary>
        public string AuthPassword { get; set; }

        /// <summary>
        /// SMTP の認証方法
        /// </summary>
        public enum SmtpAuthMethod
        {
            /// <summary>
            /// 認証なし
            /// </summary>
            None,

            /// <summary>
            /// Plain 認証
            /// </summary>
            Plain,

            /// <summary>
            /// Login 認証
            /// </summary>
            Login,

            /// <summary>
            /// CRAM-MD5 認証
            /// </summary>
            CRAM_MD5,
        }

        /// <summary>
        /// 認証方法
        /// </summary>
        public SmtpAuthMethod AuthMethod { get; set; }

        /// <summary>
        /// SMTP サーバーとの接続に SSL を使用するかどうかを指定
        /// </summary>
        public bool EnableSsl { get; set; }

        // バッファ長
        private const int _bufferLen = 1024;
        // サーバー応答のメッセージ部分の開始位置
        private const int _messageStartPos = 4;

        /// <summary>
        /// メールを送信する
        /// </summary>
        /// <exception cref="InvalidOperationException">
        /// Thrown when... プロパティに必須項目がセットされていないとき
        /// </exception>
        /// <exception cref="SocketException">
        /// Thrown when... サーバーとのコネクションが確立できなかったとき
        /// </exception>
        /// <exception cref="System.Security.Authentication.AuthenticationException">
        /// Thrown when... SSL 接続が確立できなかったとき
        /// </exception>
        /// <exception cref="ApplicationException">
        /// Thrown when... SMTPのサーバーとの交信でエラーが発生したとき
        /// </exception>
        /// <param name="mail">送信するメール</param>
        /// <returns></returns>
        public void Send(SmtpMailMessage mail)
        {
            if (ServerName == "")
                throw new InvalidOperationException("ServerName is empty.");
            if (AuthUserName == "")
                throw new InvalidOperationException("UserName is empty.");
            if (AuthPassword == "")
                throw new InvalidOperationException("Password is empty.");
            if (Encoding.GetEncodings().Where(c =>  c.Name.ToUpper() == MailEncoding.ToUpper()).Count() == 0)
                throw new InvalidOperationException("Encoding name not found(encoding: " + MailEncoding + ")");

            // サーバー接続
            Debug.WriteLine("Connecting...");
            using (var sock = new TcpClient())
            {
                try
                {
                    sock.Connect(ServerName, ServerPort);
                    Debug.WriteLine("Connected!");
                    Debug.WriteLine("Socket stream open!");
                    using (var stream = sock.GetStream())
                    {
                        try
                        {
                            if (EnableSsl)
                                // SMTP over SSL
                                sslConnect(ServerName, stream, mail);
                            else
                                // SMTP
                                smtpConnection(stream, mail);
                        }
                        finally
                        {
                            Debug.WriteLine("Socket stream close!");
                            stream.Close();
                        }
                    }
                }
                catch (System.Net.Sockets.SocketException e)
                {
                    Debug.WriteLine("Socket error! code: " + e.ErrorCode.ToString() + "; " + e.Message);
                    throw;
                }
                finally
                {
                    // サーバー接続のクローズ
                    Debug.WriteLine("Connection close!");
                    sock.Close();
                }

                return;
            }
        }

        /// <summary>
        /// SSL 接続を確立する
        /// </summary>
        /// <param name="serverName"></param>
        /// <param name="sockStream"></param>
        /// <param name="mail"></param>
        
        private void sslConnect(string serverName,
            System.IO.Stream sockStream, SmtpMailMessage mail)
        {
            using (var stream = new SslStream(sockStream))
            {
                try
                {
                    Debug.WriteLine("SSL authenticating...");
                    stream.AuthenticateAsClient(serverName);
                    Debug.WriteLine("SSL authenticated!");
                    smtpConnection(stream, mail);
                }
                catch (System.Security.Authentication.AuthenticationException e)
                {
                    Debug.WriteLine("Authentication error: " + e.Message);
                    if (e.InnerException != null)
                    {
                        Debug.WriteLine("Inner exception: " + e.InnerException.Message);
                    }
                    throw;
                }
                finally
                {
                    Debug.WriteLine("SSL stream close!");
                    stream.Close();
                }
            }
        }

        // SMTP 接続
        private void smtpConnection(System.IO.Stream stream, SmtpMailMessage mail)
        {
            const int initialDataLength = 1024; // メールデータの初期確保データ長
            string rstr;
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("220"))
                throw new ApplicationException("Server Not Ready");

            sendData(stream, "EHLO " + System.Net.Dns.GetHostName() + "\r\n");
            Debug.WriteLine("[S]EHLO " + System.Net.Dns.GetHostName());
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("250"))
                throw new ApplicationException(rstr);

            // 認証
            if (AuthMethod != SmtpAuthMethod.None)
                authenticate(stream);

            sendData(stream, "MAIL FROM:" + mail.From.Address + "\r\n");
            Debug.WriteLine("[C]MAIL FROM:" + mail.From.Address);
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("250"))
                throw new ApplicationException(rstr);

            sendData(stream, "RCPT TO:" + mail.To.Address + "\r\n");
            Debug.WriteLine("[C]RCPT TO:" + mail.To.Address);
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("250"))
                throw new ApplicationException(rstr);

            sendData(stream, "DATA\r\n");
            Debug.WriteLine("[C]DATA");
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("354"))
                throw new ApplicationException(rstr);

            // メールデータ
            var data = new StringBuilder(initialDataLength);
            data.Append("From: " + encodeDisplayAddress(mail.From) + "\r\n");
            data.Append("To: " + encodeDisplayAddress(mail.To) + "\r\n");
            if (mail.Subject != "" && !Regex.IsMatch(mail.Subject, @"^\p{IsBasicLatin}+$"))
            {   // Subject に非 ASCII 文字が含まれていたらエンコードする
                data.Append("Subject: " + string.Format("=?{0}?B?{1}?=\r\n",
                    MailEncoding, getBase64String(mail.Subject, MailEncoding)));
            }
            else
                data.Append("Subject: " + mail.Subject + "\r\n");
            var date = DateTime.Now.ToString(@"ddd, dd MMM yyyy HH\:mm\:ss",
                System.Globalization.CultureInfo.CreateSpecificCulture("en-US"));
            var timeZone = DateTimeOffset.Now.ToString("%K").Replace(":", "");
            data.Append(string.Format("Date: {0} {1}\r\n", date, timeZone));
            data.Append(string.Format("Message-ID: <{0}@{1}>\r\n", Guid.NewGuid(), mail.From.Host));
            data.Append("MIME-Version: 1.0\r\n");
            data.Append("Content-Transfer-Encoding: 7bit\r\n");
            data.Append("Content-Type: text/plain; charset=" + MailEncoding + "\r\n");
            data.Append("\r\n" +
                // see RFC 5321 4.5.2 Transparency
                Regex.Replace(mail.Body, @"^\.", "..", RegexOptions.Multiline) + "\r\n");
            data.Append(".\r\n");
            sendData(stream, data.ToString());
            Debug.WriteLine("[C]mail DATA:\r\n" + data.ToString());
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("250"))
                throw new ApplicationException(rstr);

            sendData(stream, "QUIT\r\n");
            Debug.WriteLine("[C]QUIT");
            rstr = receiveData(stream);
            Debug.WriteLine(rstr);
            if (!rstr.StartsWith("221"))
                throw new ApplicationException(rstr);
        }

        // データを受信
        private string receiveData(System.IO.Stream stream)
        {
            var enc = Encoding.GetEncoding(MailEncoding);
            var data = new byte[_bufferLen];
            int len;
            var str = "";
            using (var memStream = new System.IO.MemoryStream())
            {
                try
                {
                    // すべて受信する
                    do
                    {
                        // 届いているものを受信
                        len = stream.Read(data, 0, data.Length);
                        if (len == 0)
                            return str;     // サーバーとの接続が切断された時
                        memStream.Write(data, 0, len);
                        // デコード
                        str = enc.GetString(memStream.ToArray());
                    }
                    while (!str.EndsWith("\r\n"));
                }
                finally
                {
                    memStream.Close();
                }
            }

            return str;
        }

        // データを送信
        private void sendData(System.IO.Stream stream, string str)
        {
            var enc = Encoding.GetEncoding(MailEncoding);

            // byte 型配列に変換
            var data = enc.GetBytes(str);
            stream.Write(data, 0, data.Length);
        }

        // 認証
        private void authenticate(System.IO.Stream stream)
        {
            string rstr;

            switch (AuthMethod)
            {
                case SmtpAuthMethod.Plain:
                    sendData(stream, "AUTH PLAIN\r\n");
                    Debug.WriteLine("[S]AUTH PLAIN");
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("334"))
                    {
                        if (rstr.StartsWith("502"))
                        {   // 502:Command not implemented; SMTP AUTH をサポートしていない
                            Debug.WriteLine(rstr);
                            return;
                        }
                        // 504:Command parameter not implemented; AUTH PLAIN をサポートしていない
                        // サーバーからそれなりのエラーメッセージが期待できるので、そのままメッセージとして活用する
                        throw new ApplicationException(rstr);
                    }

                    string str = AuthUserName + '\0' + AuthUserName + '\0' + AuthPassword;
                    sendData(stream, getBase64String(str) + "\r\n");
                    Debug.WriteLine("[C]base64: " + getBase64String(str));
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("235"))
                        throw new ApplicationException(rstr);
                    break;

                case SmtpAuthMethod.Login:
                    sendData(stream, "AUTH LOGIN\r\n");
                    Debug.WriteLine("[C]AUTH LOGIN");
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("334"))
                    {
                        if (rstr.StartsWith("502"))
                        {   // 502:Command not implemented; SMTP AUTH をサポートしていない
                            Debug.WriteLine(rstr);
                            return;
                        }
                        // 504:Command parameter not implemented; AUTH LOGIN をサポートしていない
                        throw new ApplicationException(rstr);
                    }

                    sendData(stream, getBase64String(AuthUserName) + "\r\n");
                    Debug.WriteLine("[C]" + AuthUserName + "; base64: " + getBase64String(AuthUserName));
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("334"))
                        throw new ApplicationException(rstr);

                    sendData(stream, getBase64String(AuthPassword) + "\r\n");
                    Debug.WriteLine("[C]" + AuthPassword + "; base64: " + getBase64String(AuthPassword));
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("235"))
                        throw new ApplicationException(rstr);
                    break;

                case SmtpAuthMethod.CRAM_MD5:
                    sendData(stream, "AUTH CRAM-MD5\r\n");
                    Debug.WriteLine("[C]AUTH CRAM-MD5");
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("334"))
                    {
                        if (rstr.StartsWith("502"))
                        {   // 502:Command not implemented; SMTP AUTH をサポートしていない
                            Debug.WriteLine(rstr);
                            return;
                        }
                        // 504:Command parameter not implemented; AUTH CRAM-MD5 をサポートしていない
                        throw new ApplicationException(rstr);
                    }
                    var key = (new System.Security.Cryptography.HMACMD5(Encoding.UTF8.GetBytes(AuthPassword)))
                        .ComputeHash(System.Convert.FromBase64String(rstr.Substring(_messageStartPos)));
                    var digest = "";
                    foreach (var octet in key)
                    {
                        digest += octet.ToString("x02");
                    }
                    var response = getBase64String(string.Format("{0} {1}", AuthUserName, digest));
                    sendData(stream, response + "\r\n");
                    Debug.WriteLine("[C]base64: " + response);
                    rstr = receiveData(stream);
                    Debug.WriteLine(rstr);
                    if (!rstr.StartsWith("235"))
                        throw new ApplicationException(rstr);
                    break;
            }
        }

        // 指定された文字コードでエンコードし、さらに Base64 でエンコード
        private string getBase64String(string str, string charSetName = "utf-8")
        {
            var enc = Encoding.GetEncoding(charSetName);
            return Convert.ToBase64String(enc.GetBytes(str));
        }

        // メールアドレスの表記名部分が非 ASCII なときにはエンコード
        private System.Net.Mail.MailAddress encodeDisplayAddress(System.Net.Mail.MailAddress email)
        {
            if (email.DisplayName != "" && !Regex.IsMatch(email.DisplayName, @"^\p{IsBasicLatin}+$"))
            {
                var enc = Encoding.GetEncoding(MailEncoding);
                var dispName = string.Format("=?{0}?B?{1}?=", MailEncoding,
                    getBase64String(email.DisplayName, MailEncoding));
                var mail = new System.Net.Mail.MailAddress(email.Address, dispName);
                return mail;
            }
            return email;
        }
    }
}
