﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace TcpComTest.Server
{
    class Server
    {
        // Protocol
        // (クライアント接続)
        // [100 Service ready\r\n] を送信
        // [HELO\r\n] を受信
        // [110 Request completed\r\n] を送信
        // [GET\r\n]を受信
        // [120 Send data\r\n] を送信
        // (データを送信 終端は\x1a\r\n) ... \x1A : EOF
        // [QUIT\r\n]を受信
        // [110 Request completed\r\n]を送信
        // (クライアント接続を閉じる)
        //
        // 接続待ちがなければ終了
        //
        // その他のレスポンス・コード
        // [500 Syntax error]
        // [501 Bad sequence of commands]
        // [502 Timeout connection closed]

        private enum serverStatus
        {
            None,
            TRANSACTION
        }

        private static serverStatus status;

        static void Main(string[] args)
        {
            // 文字コードを UTF-8 とする
            var enc = System.Text.Encoding.UTF8;

            status = Server.serverStatus.None;

            // ローカルアドレスで　Listen を開始する
            var host = "localhost";
            int port = 2001;
            var ipAddress = System.Net.Dns.GetHostAddresses(host).First();
            System.Net.Sockets.TcpListener listener = null;
            try
            {
                listener = new System.Net.Sockets.TcpListener(ipAddress, port);
                listener.Start();
                Console.WriteLine("Port[{0}]の Listen を開始しました。", port);

                do
                {
                    // 接続要求があったら受け入れる
                    using (var tcp = listener.AcceptTcpClient())
                    {
                        Console.WriteLine("クライアントが接続しました。");
                        // NetworkStream を取得
                        using (var stream = tcp.GetStream())
                        {
                            try
                            {
                                // Read のタイムアウトに15秒を設定
                                (stream as System.Net.Sockets.NetworkStream).ReadTimeout = 15000;

                                sendData(stream, "100 Service ready\r\n");

                                var rString = receiveData(stream, enc);
                                if (rString == "HELO\r\n")
                                {
                                    status = Server.serverStatus.TRANSACTION;
                                    sendData(stream, "110 Request completed\r\n");
                                    transaction(stream, enc);
                                }
                                else
                                {
                                    sendData(stream, "501 Bad sequence of commands\r\n");
                                }
                            }
                            catch (System.IO.IOException e)
                            {
                                if (e.InnerException.GetType() == typeof(System.Net.Sockets.SocketException) &&
                                    (e.InnerException as System.Net.Sockets.SocketException).ErrorCode == 10060)
                                {   // タイムアウト発生時
                                    sendData(stream, "502 Timeout connection closed\r\n");
                                    Console.WriteLine("タイムアウトが発生したため、クライアントとの接続を切断します。");
                                }
                                else
                                {
                                    throw;
                                }
                            }

                            stream.Close();
                        }

                        tcp.Close();
                        Console.WriteLine("切断しました。");
                    }
                } while (listener.Pending());
            }
            catch (System.Net.Sockets.SocketException e)
            {
                Console.WriteLine("SocketException: {0}", e);
            }
            finally
            {
                listener.Stop();
            }

            Console.ReadLine();
        }

        private static void transaction(System.IO.Stream stream, System.Text.Encoding enc)
        {
            string rString = "";
            try
            {
                do
                {
                    if (status != Server.serverStatus.TRANSACTION)
                    {
                        throw new InvalidOperationException("ステータスが TRANSACTION でないのでコマンドは処理できません。");
                    }

                    rString = receiveData(stream, enc);
                    switch (rString)
                    {
                        case "QUIT\r\n":
                            quit(stream);
                            break;
                        case "GET\r\n":
                            get(stream);
                            break;
                        default:
                            invalidCommand(stream);
                            break;
                    }
                }
                while (rString != "QUIT\r\n");
            }
            catch (ClientCloseException)
            {
            }

            status = Server.serverStatus.None;
        }

        private static void quit(System.IO.Stream stream)
        {
            sendData(stream, "110 Request completed\r\n");
        }

        private static void get(System.IO.Stream stream)
        {
            var data = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789\r\n" +
                       "あいうえお\r\nかきくけこ\r\nさしすせそ\r\nたちつてと\r\nなにぬねの\r\n" +
                       "はひふへほ\r\nまみむめも\r\nやゆよ\r\nらりるれろ\r\nわを\r\nん\r\n";

            sendData(stream, "120 Send data\r\n");
            sendData(stream, data + "\x1a\r\n");
        }

        private static void invalidCommand(System.IO.Stream stream)
        {
            sendData(stream, "500 Syntax error\r\n");
        }

        private static string receiveData(System.IO.Stream stream, System.Text.Encoding enc)
        {
            // クライアントから送られてきたデータを受信する
            var recieveMessage = "";
            using (var memStream = new System.IO.MemoryStream())
            {
                var rBuff = new byte[256];
                int rSize;
                do
                {
                    // データの一部を受信する
                    rSize = stream.Read(rBuff, 0, rBuff.Length);
                    // rSize が 0 のときにはクライアントが切断したと判断
                    if (rSize == 0)
                    {
                        Console.WriteLine("クライアントが切断しました。");
                        throw new ClientCloseException("ストリームの読み出しの際、クライアントが切断していました。");
                    }
                    // 受信したデータを蓄積する
                    memStream.Write(rBuff, 0, rSize);
                } while ((stream as System.Net.Sockets.NetworkStream).DataAvailable);

                // 受信したデータを文字列に変換
                recieveMessage = enc.GetString(memStream.ToArray());
                memStream.Close();
            }

            Console.WriteLine("[C]" + recieveMessage);
            return recieveMessage;
        }

        private static void sendData(System.IO.Stream stream, string str)
        {
            // 文字列をバイト配列へ
            var data = System.Text.Encoding.UTF8.GetBytes(str);
            stream.Write(data, 0, data.Length);
        }
    }
}
